-- FILE: \DustParticles.lua
function AddDustParticles()
	-- ######################################################
	-- ################### Customizing ######################
	-- ######################################################
	
	-- Define your static slots in which objects you want particles to appear (look for ID / Number of Slot in WadTool), further with comma seperated, do not remove local staticStartSlots{}
	-- if you have no static object, leave this variable like this:
	-- local staticStartSlots = {}
	local staticStartSlots = {
		114
	}
	
	-- Define your moveable slots in which objects where you want particles to appear (look for ID / Number of Slot in WadTool), further with comma seperated, do not remove local moveableStartSlots{}
	-- if you have no moveable object, leave this variable like this:
	-- local moveableStartSlots = {}
	local moveableStartSlots = {
		1020
	}
	
	-- Define the range where the particles should randomly appear from the object (1024 = one Square)
	local xPosTolerance = 1024
	local yPosTolerance = 1024
	local zPosTolerance = 1024
	
	-- Define Random Velocity Tolerance of particles. Larger values speed up your particles
	local xSpeed = 2
	local ySpeed = 2
	local zSpeed = 2
	
	-- Customizing Random Density (larger values generate more particles per frame)
	local density = 3
	
	-- Sprite Index (see index of a sprite in DEFAULT_SPRITES object.)
	local spriteIndex = 1
	
	-- Set random Gravity tolerance, larger values ​​cause the particles to rise or fall more quickly
	local gravityTolerance = 1
	
	-- Random Rotation
	local rotationTolerance = 1
	
	-- Set the color that the particle should initially have after spawn (0-255) 
	local colorFromRed = 255
	local colorFromGreen = 125
	local colorFromBlue = 50
	
	-- Set the color that the particle should have at the end (0-255)
	local colorToRed = 255
	local colorToGreen = 255
	local colorToBlue = 255
	
	-- Set Blend Mode
	-- Use following Modes: https://tombengine.github.io/4%20enums/Effects.BlendID.html
	-- Best for dust particles is ADDITIVE
	local mode = TEN.Effects.BlendID.ADDITIVE
	
	-- Random Start Size (from 1-x), 10 is approx as big as Lara's head, set more than 2 if you would like to have real dust bunnies ;-)
	local startSizeTolerance = 3
	
	-- Set random End Size (from 1-x), 10 is approx as big as Lara's head, set more than 2 if you would like to have real dust bunnies ;-)
	local endSizeTolerance = 3
	
	-- Set random Lifetime of particle (1-x) (in seconds), larger values generate more particles after time
	local lifeTime = 15
	
	-- Set it to true if you want Lara to be hurt by touch
	local damage = false
	
	-- Set it to true if you want Lara to be poison by touch
	local poison = false
	
	-- Max Distance from Lara where Particles will render
	local distanceFromLara = 6144
	
	-- ######################################################
	-- ########### End Customizing ##########################
	-- ######################################################

	local objectsForParticles = {}
	
	for index, staticStartSlot in pairs(staticStartSlots) do
		local statics = GetStaticsBySlot(staticStartSlot)
    	for index, static in pairs(statics) do
    		if CalculateDistance(static:GetPosition(), Lara:GetPosition()) < distanceFromLara then
        		table.insert(objectsForParticles, static)
        	end
    	end
    end
    
    for index, moveableStartSlot in pairs(moveableStartSlots) do
		local moveables = GetMoveablesBySlot(moveableStartSlot)
    	for index, moveable in pairs(moveables) do
    		if CalculateDistance(moveable:GetPosition(), Lara:GetPosition()) < distanceFromLara then
        		table.insert(objectsForParticles, moveable)
        	end
    	end
    end
    
    for index, objectForParticle in pairs(objectsForParticles) do
        --print(objectForParticle:GetName())
        -- Calculate start positions
		local vec3Pos = objectForParticle:GetPosition();
		
		local xPos = vec3Pos.x
		local yPos = vec3Pos.y
		local zPos = vec3Pos.z
		
		if xPosTolerance > 0 then
			xPos = vec3Pos.x+math.random(xPosTolerance)-math.random(xPosTolerance)
		end
		
		if yPosTolerance > 0 then
			yPos = vec3Pos.y+math.random(yPosTolerance)-math.random(yPosTolerance)
		end
		
		if zPosTolerance > 0 then
			zPos = vec3Pos.z+math.random(zPosTolerance)-math.random(zPosTolerance)
		end
		
		-- Calculate Velocity
		local xVelo = 0 + math.random(xSpeed) - math.random(xSpeed)
		local yVelo = 0 + math.random(ySpeed) - math.random(ySpeed)
		local zVelo = 0 + math.random(zSpeed) - math.random(zSpeed)
		
		for i=1,math.random(1,density) do 
			EmitParticle(
				Vec3(xPos, yPos, zPos), --start position
				Vec3(xVelo, yVelo, zVelo), -- velocity
				spriteIndex, -- spriteIndex
				math.random(0-gravityTolerance, 0+gravityTolerance), -- gravity
				math.random(0-rotationTolerance, 0+rotationTolerance), -- rotation
				Color(colorFromRed, colorFromGreen, colorFromBlue), -- startColor
				Color(colorToRed, colorToGreen, colorToBlue), -- endColor
				mode, -- blendMode
				math.random(startSizeTolerance), -- startSize
				math.random(endSizeTolerance), -- endSize
				math.random(lifeTime), -- lifetime
				damage, -- damage
				poison -- poison
			)
		end
	end
end