/*
 *  Copyright (C) 2002-2010  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../dos/drives.h"
#include "control.h"
#include "cpu.h"
#include "render.h"
#include "menu.h"
#include "glidedef.h"
#include "SDL.h"
#include "SDL_syswm.h"

void SetVal(const std::string secname, std::string preval, const std::string val) {
	if(preval=="keyboardlayout") {
		DOS_MCB mcb(dos.psp()-1);
		static char name[9];
		mcb.GetFileName(name);
		if (strlen(name)) {
			LOG_MSG("GUI: Exit %s running in DOSBox, and then try again.",name);
			return;
		}
	}
	Section* sec = control->GetSection(secname);
	sec->ExecuteDestroy(false);
	if(sec) {
        std::string real_val=preval+"="+val;
		sec->HandleInputline(real_val);
		sec->ExecuteInit(false);
	}
}

MENU_Block menu;

#ifdef WIN32
#include <shlobj.h>

extern void RENDER_CallBack( GFX_CallBackFunctions_t function );

unsigned int hdd_defsize=16000;
char hdd_size[20]="";

HWND GetHWND(void) {
	SDL_SysWMinfo wmi;
	SDL_VERSION(&wmi.version);

	if(!SDL_GetWMInfo(&wmi)) {
		return NULL;
		//LOG_MSG("SDL:Error retrieving window information");
		//E_Exit("Failed to get window info");
	}
	return wmi.window;
}

void GetDefaultSize(void) {
	char sizetemp[20]="512,32,32765,";
	char sizetemp2[20]="";
	sprintf(sizetemp2,"%d",hdd_defsize);
	strcat(sizetemp,sizetemp2);
	sprintf(hdd_size,sizetemp);
}

void SearchFolder( char path[MAX_PATH], char drive, std::string drive_type ) {
	WIN32_FIND_DATA FindFileData;
	HANDLE hFind;

	hFind = FindFirstFile ( "*.*", &FindFileData );

	if ( hFind != INVALID_HANDLE_VALUE ) MountDrive_2(drive,path,drive_type);
	FindClose ( hFind );
}

void BrowseFolder( char drive , std::string drive_type ) {
	if (Drives[drive-'A']) {
		LOG_MSG("Unmount drive %c first, and then try again.",drive);
		return;
	}

	char path[MAX_PATH];
	BROWSEINFO bi = { 0 };
	if(drive_type=="CDROM")
		bi.lpszTitle = ("Select a drive/directory to mount as CD-ROM\nMounting a directory as CD-ROM gives an limited support");
	else if(drive_type=="FLOPPY")
		bi.lpszTitle = ("Select a drive/directory to mount as Floppy");
	else if(drive_type=="LOCAL")
		bi.lpszTitle = ("Select a drive/directory to mount as Local");
	else
		bi.lpszTitle = ("Select a drive/directory to mount.");
	LPITEMIDLIST pidl = SHBrowseForFolder ( &bi );

	if ( pidl != 0 ) {
		SHGetPathFromIDList ( pidl, path );
//		SetCurrentDirectory ( path );
		SearchFolder( path , drive, drive_type );
		IMalloc * imalloc = 0;
		if ( SUCCEEDED( SHGetMalloc ( &imalloc )) ) {
			imalloc->Free ( pidl );
			imalloc->Release ( );
		}
	}
}

/*
void cycle_conf(void) {
	if (CPU_CycleAutoAdjust) return;
	char tmp[10];
	if (CPU_AutoDetermineMode) {
		if(CPU_CyclePercUsed != 100)
			sprintf(tmp,"auto %d %d%%",CPU_CycleMax,CPU_CyclePercUsed);
		else
			sprintf(tmp,"auto %d",CPU_CycleMax);
	} else
		sprintf(tmp,"%d",CPU_CycleMax);
	Section* sec = control->GetSection("cpu");
	char cycles_set[10];
	if(sec) {
		strcpy(cycles_set,tmp);
		sec->ExecuteDestroy(false);
		std::string tmp2("cycles="); tmp2.append(cycles_set); sec->HandleInputline(tmp2);
		sec->ExecuteInit(false);
	}
}
*/

void mem_conf(std::string memtype, int option) {
	char tmp[10];
	Section* sec = control->GetSection("dos");
	Section_prop * section=static_cast<Section_prop *>(sec); 
	if(!option) sprintf(tmp, section->Get_bool(memtype) ? "false":"true");
		else {
			switch (option) {
				case 1: sprintf(tmp, "true"); break;
				case 2: sprintf(tmp, "false"); break;
				case 3: sprintf(tmp, "emsboard"); break;
				case 4: sprintf(tmp, "emm386"); break;
				default: return;
			}
		}
	if(sec) {
		sec->ExecuteDestroy(false);
		std::string tmp2(memtype);
		tmp2.append("="); tmp2.append(tmp);
		sec->HandleInputline(tmp2);
		sec->ExecuteInit(false);
	}
}

void UnMount(int i_drive) {
	i_drive = toupper(i_drive);
	if(i_drive-'A' == DOS_GetDefaultDrive()) {
		DOS_MCB mcb(dos.psp()-1);
		static char name[9];
		mcb.GetFileName(name);
		if (!strlen(name)) goto umount;
		LOG_MSG("GUI:Drive %c is being used. Aborted.",i_drive); return;
		return;
	};
umount:
	if (i_drive-'A' < DOS_DRIVES && i_drive-'A' >= 0 && Drives[i_drive-'A']) {
		switch (DriveManager::UnmountDrive(i_drive-'A')) {
		case 0:
			Drives[i_drive-'A'] = 0;
			if(i_drive-'A' == DOS_GetDefaultDrive()) 
				DOS_SetDrive(toupper('Z') - 'A');
			LOG_MSG("GUI:Drive %c has succesfully been removed.",i_drive); break;
		case 1:
			LOG_MSG("GUI:Virtual Drives can not be unMOUNTed."); break;
		case 2:
			LOG_MSG("GUI:MSCDEX Failure: Drive-letters of multiple CDRom-drives have to be continuous."); break;
		}
	}
}

#include "../dos/cdrom.h"
extern void MSCDEX_SetCDInterface(int intNr, int numCD);
void MountDrive_2(char drive, const char drive2[DOS_PATHLENGTH], std::string drive_type) {
	DOS_Drive * newdrive;
	std::string temp_line;
	std::string str_size;
	Bit16u sizes[4];
	Bit8u mediaid;
	//struct stat test;
	int num = SDL_CDNumDrives();

	if((drive_type=="LOCAL") && (drive2=="C:\\")) {
		char drive_warn[70];
		sprintf(drive_warn,"Mounting C:\\ is NOT recommended.\nDo you want to continue?");
		if (MessageBox(GetHWND(),drive_warn,"Warning",MB_YESNO)==IDNO) return;
	}

	if(drive_type=="CDROM") {
		mediaid=0xF8;		/* Hard Disk */
		str_size="650,127,16513,1700";
	} else {
		if(drive_type=="FLOPPY") {
			str_size="512,1,2847,2847";	/* All space free */
			mediaid=0xF0;			/* Floppy 1.44 media */
		} else if(drive_type=="LOCAL") {
			mediaid=0xF8;
			//extern void GetDefaultSize(void);
			//extern char hdd_size[20];
			GetDefaultSize();
			str_size=hdd_size;		/* Hard Disk */
		}
	}

	char number[20]; const char * scan=str_size.c_str();
	Bitu index=0; Bitu count=0;
		while (*scan) {
			if (*scan==',') {
				number[index]=0;sizes[count++]=atoi(number);
				index=0;
			} else number[index++]=*scan;
			scan++;
		}
	number[index]=0; sizes[count++]=atoi(number);

	temp_line = drive2;
	if(temp_line.size() > 3 && temp_line[temp_line.size()-1]=='\\') temp_line.erase(temp_line.size()-1,1);
	if (temp_line[temp_line.size()-1]!=CROSS_FILESPLIT) temp_line+=CROSS_FILESPLIT;
	Bit8u bit8size=(Bit8u) sizes[1];

	if(drive_type=="CDROM") {
		num = -1;
		int error;

		int id, major, minor;
		DOSBox_CheckOS(id, major, minor);
		if ((id==VER_PLATFORM_WIN32_NT) && (major>5)) {
			// Vista/above
			MSCDEX_SetCDInterface(CDROM_USE_IOCTL_DX, num);
		} else {
			MSCDEX_SetCDInterface(CDROM_USE_IOCTL_DIO, num);
		}
		newdrive  = new cdromDrive(drive,temp_line.c_str(),sizes[0],bit8size,sizes[2],0,mediaid,error);
			switch (error) {
				case 0  :	LOG_MSG("GUI: MSCDEX installed.");	break;
				case 1  :	LOG_MSG("GUI: MSCDEX Failure: Drive-letters of multiple CDRom-drives have to be continuous.");	break;
				case 2  :	LOG_MSG("GUI: MSCDEX Failure: Not yet supported.");	break;
				case 3  :	LOG_MSG("GUI: MSCDEX Failure: Path not valid.");				break;
				case 4  :	LOG_MSG("GUI: MSCDEX Failure: Too many CDRom-drives (max: 5). MSCDEX Installation failed");		break;
				case 5  :	LOG_MSG("GUI: MSCDEX Mounted subdirectory: limited support.");		break;
				default :	LOG_MSG("GUI: MSCDEX Failure: Unknown error");			break;
			};
	} else newdrive=new localDrive(temp_line.c_str(),sizes[0],bit8size,sizes[2],sizes[3],mediaid);

	if (!newdrive) E_Exit("DOS:Can't create drive");
	Drives[drive-'A']=newdrive;
	mem_writeb(Real2Phys(dos.tables.mediaid)+(drive-'A')*2,mediaid);
	if(drive_type=="CDROM")
		LOG_MSG("GUI: Drive %c is mounted as CD-ROM",drive);
	else
		LOG_MSG("GUI: Drive %c is mounted as local directory",drive);
    if(drive == drive2[0] && sizeof(drive2) == 4) {
        // automatic mount
    } else {
        if(drive_type=="CDROM") return;
        std::string label;
        label = drive;
        if(drive_type=="LOCAL")
            label += "_DRIVE";
        else
            label += "_FLOPPY";
        newdrive->SetLabel(label.c_str(),false,true);
    }
}

void MountDrive(char drive, const char drive2[DOS_PATHLENGTH]) {
	DOS_Drive * newdrive;
	char drive_warn[70];
	std::string temp_line;
	std::string str_size;
	Bit16u sizes[4];
	Bit8u mediaid;
	//struct stat test;
	int num = SDL_CDNumDrives();
	if(GetDriveType(drive2)==DRIVE_CDROM) {
		sprintf(drive_warn,"Do you really want to give DOSBox access to your real CD-ROM drive %c?",drive);
	}
	else if(GetDriveType(drive2)==DRIVE_REMOVABLE) {
		sprintf(drive_warn,"Do you really want to give DOSBox access to your real floppy drive %c?",drive);
	}
	else {
		sprintf(drive_warn,"Do you really want to give DOSBox access to everything\non your real drive %c?",drive);
	}

	if (MessageBox(GetHWND(),drive_warn,"Warning",MB_YESNO)==IDNO) return;

	if((GetDriveType(drive2)==DRIVE_FIXED) && (strcasecmp(drive2,"C:\\")==0)) {
		sprintf(drive_warn,"Mounting C:\\ is NOT recommended.\nDo you want to continue?");
		if (MessageBox(GetHWND(),drive_warn,"Warning",MB_YESNO)==IDNO) return;
	}

	if(GetDriveType(drive2)==DRIVE_CDROM) {
		mediaid=0xF8;		/* Hard Disk */
		str_size="650,127,16513,1700";
	} else {
		if(GetDriveType(drive2)==DRIVE_REMOVABLE) {
			str_size="512,1,2847,2847";	/* All space free */
			mediaid=0xF0;			/* Floppy 1.44 media */
		} else {
			mediaid=0xF8;
				//extern void GetDefaultSize(void);
				//extern char hdd_size[20];
				GetDefaultSize();
				str_size=hdd_size;	/* Hard Disk */
		}
	}

	char number[20]; const char * scan=str_size.c_str();
	Bitu index=0; Bitu count=0;
		while (*scan) {
			if (*scan==',') {
				number[index]=0;sizes[count++]=atoi(number);
				index=0;
			} else number[index++]=*scan;
			scan++;
		}
	number[index]=0; sizes[count++]=atoi(number);
	Bit8u bit8size=(Bit8u) sizes[1];

	temp_line = drive2;
	int error; num = -1;
	if(GetDriveType(drive2)==DRIVE_CDROM) {
		int id, major, minor;
		DOSBox_CheckOS(id, major, minor);

		if ((id==VER_PLATFORM_WIN32_NT) && (major>5)) {
			// Vista/above
			MSCDEX_SetCDInterface(CDROM_USE_IOCTL_DX, num);
		} else {
			MSCDEX_SetCDInterface(CDROM_USE_IOCTL_DIO, num);
		}
		newdrive  = new cdromDrive(drive,temp_line.c_str(),sizes[0],bit8size,sizes[2],0,mediaid,error);
			switch (error) {
				case 0  :	LOG_MSG("GUI: MSCDEX installed.");	break;
				case 1  :	LOG_MSG("GUI: MSCDEX Failure: Drive-letters of multiple CDRom-drives have to be continuous.");	break;
				case 2  :	LOG_MSG("GUI: MSCDEX Failure: Not yet supported.");	break;
				case 3  :	LOG_MSG("GUI: MSCDEX Failure: Path not valid.");				break;
				case 4  :	LOG_MSG("GUI: MSCDEX Failure: Too many CDRom-drives (max: 5). MSCDEX Installation failed");		break;
				case 5  :	LOG_MSG("GUI: MSCDEX Mounted subdirectory: limited support.");		break;
				default :	LOG_MSG("MSCDEX: Failure: Unknown error");			break;
			};
	} else newdrive=new localDrive(temp_line.c_str(),sizes[0],bit8size,sizes[2],sizes[3],mediaid);

	if (!newdrive) E_Exit("DOS:Can't create drive");
	if(error && (GetDriveType(drive2)==DRIVE_CDROM)) return;
	Drives[drive-'A']=newdrive;
	mem_writeb(Real2Phys(dos.tables.mediaid)+(drive-'A')*2,mediaid);
	if(GetDriveType(drive2)==DRIVE_CDROM) LOG_MSG("GUI: Drive %c is mounted as CD-ROM %c:\\",drive,drive);
	else LOG_MSG("GUI: Drive %c is mounted as local directory %c:\\",drive,drive);
    if(drive == drive2[0] && sizeof(drive2) == 4) {
        // automatic mount
    } else {
        if(GetDriveType(drive2) == DRIVE_CDROM) return;
        std::string label;
        label = drive;
        if(GetDriveType(drive2) == DRIVE_FIXED)
            label += "_DRIVE";
        else
            label += "_FLOPPY";
        newdrive->SetLabel(label.c_str(),false,true);
    }
}

void Mount_Zip(char drive, std::string temp_line) {
	DOS_Drive * newdrive;
	std::string str_size;
	Bit16u sizes[4];
	Bit8u mediaid;
	//struct stat test;
	mediaid=0xF8;

	GetDefaultSize();
	str_size=hdd_size;

	char number[20];const char * scan=str_size.c_str();
	Bitu index=0;Bitu count=0;
	/* Parse the str_size string */
	while (*scan) {
		if (*scan==',') {
			number[index]=0;sizes[count++]=atoi(number);
			index=0;
		} else number[index++]=*scan;
		scan++;
	}
	number[index]=0;sizes[count++]=atoi(number);

	temp_line.insert(0, 1, ':');
	temp_line += "\\";
	if (temp_line.size() > 3 && temp_line[temp_line.size()-1]=='\\') temp_line.erase(temp_line.size()-1,1);
	if (temp_line[temp_line.size()-1]!=CROSS_FILESPLIT) temp_line+=CROSS_FILESPLIT;
	Bit8u bit8size=(Bit8u) sizes[1];

#if C_HAVE_PHYSFS
	newdrive=new physfsDrive(temp_line.c_str(),sizes[0],bit8size,sizes[2],sizes[3],mediaid);
#else
	newdrive = 0;
	LOG_MSG("ERROR:This build does not support physfs");
#endif

	if (!newdrive) E_Exit("DOS:Can't create drive");
	Drives[drive-'A']=newdrive;
	mem_writeb(Real2Phys(dos.tables.mediaid)+(drive-'A')*2,newdrive->GetMediaByte());
	LOG_MSG("%s",newdrive->GetInfo());
	LOG_MSG("Drive %c is mounted as PHYSFS directory",drive);
}

void Mount_Img(char drive, std::string realpath) {
	DOS_Drive * newdrive = NULL;
        imageDisk * newImage = NULL;
	//Bit32u imagesize;
	std::string label;
	std::string temp_line = realpath;
	std::vector<std::string> paths;

	Bit8u mediaid;
	Bit16u sizes[4];
	std::string str_size;
	mediaid=0xF8;
	str_size="650,127,16513,1700";
	mediaid=0xF8;
	char number[20];
	const char * scan=str_size.c_str();
	Bitu index=0;Bitu count=0;
	while (*scan) {
		if (*scan==',') {
			number[index]=0;sizes[count++]=atoi(number);
			index=0;
		} else number[index++]=*scan;
		scan++;
	}
	number[index]=0;sizes[count++]=atoi(number);
	struct stat test;
	if (stat(temp_line.c_str(),&test)) {
		// convert dosbox filename to system filename
		char fullname[CROSS_LEN];
		char tmp[CROSS_LEN];
		safe_strncpy(tmp, temp_line.c_str(), CROSS_LEN);
		Bit8u dummy;
		localDrive *ldp = dynamic_cast<localDrive*>(Drives[dummy]);
		ldp->GetSystemFilename(tmp, fullname);
		temp_line = tmp;
	}
	paths.push_back(temp_line);
	if (paths.size() == 1)
		temp_line = paths[0];
	MSCDEX_SetCDInterface(CDROM_USE_SDL, -1);
	// create new drives for all images
	std::vector<DOS_Drive*> isoDisks;
	std::vector<std::string>::size_type i;
	for (i = 0; i < paths.size(); i++) {
		int error = -1;
		DOS_Drive* newDrive = new isoDrive(drive, paths[i].c_str(), mediaid, error);
		isoDisks.push_back(newDrive);
		switch (error) {
			case 0  :	break;
			case 1  :	LOG_MSG("GUI: MSCDEX Failure: Drive-letters of multiple CDRom-drives have to be continuous.");	break;
			case 2  :	LOG_MSG("GUI: MSCDEX Failure: Not yet supported.");	break;
			case 3  :	LOG_MSG("GUI: MSCDEX Failure: Path not valid.");				break;
			case 4  :	LOG_MSG("GUI: MSCDEX Failure: Too many CDRom-drives (max: 5). MSCDEX Installation failed.");		break;
			case 5  :	LOG_MSG("GUI: MSCDEX Mounted subdirectory: limited support.");		break;
			case 6  :	LOG_MSG("GUI: MSCDEX Failure: File is either no iso/cue image or contains errors.");		break;
			default :	LOG_MSG("GUI: MSCDEX Failure: Unknown error.");			break;
	}
	// error: clean up and leave
	if (error) {
		for(i = 0; i < isoDisks.size(); i++) {
			delete isoDisks[i];
		}
		return;
	}
	// Update DriveManager
	for(i = 0; i < isoDisks.size(); i++) {
		DriveManager::AppendDisk(drive - 'A', isoDisks[i]);
	}
	DriveManager::InitializeDrive(drive - 'A');

	// Set the correct media byte in the table 
	mem_writeb(Real2Phys(dos.tables.mediaid) + (drive - 'A') * 2, mediaid);

	// Print status message (success)
	LOG_MSG("MSCDEX installed.");
	std::string tmp(paths[0]);
	for (i = 1; i < paths.size(); i++) {
		tmp += "; " + paths[i];
	}
	LOG_MSG("GUI: Drive %c is mounted as %s", drive, tmp.c_str());

	// check if volume label is given
	//if (cmd->FindString("-label",label,true)) newdrive->dirCache.SetLabel(label.c_str());
	return;
	}
}

void DOSBox_SetMenu(void) {
	if(!menu.gui) return;
	menu.toggle=true;
	SetMenu(GetHWND(), LoadMenu(GetModuleHandle(NULL),MAKEINTRESOURCE(IDR_MENU)));
	DrawMenuBar (GetHWND());

	if (glide.enabled)
		GLIDE_ResetScreen();
	else {
		if(menu.startup) {
			RENDER_CallBack( GFX_CallBackReset );
		}
	}
}

void DOSBox_NoMenu(void) {
	if(!menu.gui) return;
	menu.toggle=false;
	SetMenu(GetHWND(), NULL);
	DrawMenuBar(GetHWND());
	if (glide.enabled)
		GLIDE_ResetScreen();
	else {
	RENDER_CallBack( GFX_CallBackReset );
    }
}

void DOSBox_CheckOS(int &id, int &major, int &minor) {
	OSVERSIONINFO osi;
	ZeroMemory(&osi, sizeof(OSVERSIONINFO));
	osi.dwOSVersionInfoSize = sizeof(osi);
	GetVersionEx(&osi);
	id=osi.dwPlatformId;
	if(id==1) { major=0; minor=0; return; }
	major=osi.dwMajorVersion;
	minor=osi.dwMinorVersion;
}

bool DOSBox_Kor(void) {
    if(menu.compatible) return false;
    char Buffer[30];
    GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SABBREVLANGNAME, Buffer, sizeof(Buffer));
    return (!strcmp(Buffer,"KOR") ? true : false);
}

void DOSBox_RefreshMenu(void) {
    int width, height; bool fullscreen;
    void GFX_GetSize(int &width, int &height, bool &fullscreen);
    GFX_GetSize(width,height,fullscreen);
    void SDL_Prepare(void);
    SDL_Prepare();
    if(!menu.gui) return;

    if(fullscreen && !glide.enabled) {
    	SetMenu(GetHWND(), NULL);
    	DrawMenuBar(GetHWND());
        return;
    }
	if(menu.toggle)
		DOSBox_SetMenu();
	else
		DOSBox_NoMenu();
}

void DOSBox_RefreshMenu2(void) {
	if(!menu.gui) return;
   int width, height; bool fullscreen;
   void GFX_GetSize(int &width, int &height, bool &fullscreen);
   GFX_GetSize(width,height,fullscreen);
    void SDL_Prepare(void);
    SDL_Prepare();
    if(!menu.gui) return;

    if(fullscreen && !glide.enabled) {
    	SetMenu(GetHWND(), NULL);
    	DrawMenuBar(GetHWND());
        return;
    }
	if(menu.toggle) {
		menu.toggle=true;
		SetMenu(GetHWND(), LoadMenu(GetModuleHandle(NULL),MAKEINTRESOURCE(IDR_MENU)));
		DrawMenuBar (GetHWND());
	} else {
		menu.toggle=false;
		SetMenu(GetHWND(), NULL);
		DrawMenuBar(GetHWND());
	}
}

void ToggleMenu(bool pressed) {
    menu.resizeusing=true;
	int width, height; bool fullscreen;
	void GFX_GetSize(int &width, int &height, bool &fullscreen);
	GFX_GetSize(width, height, fullscreen);
    if(!menu.gui || !pressed || fullscreen) return;
	if(!menu.toggle) {
		menu.toggle=true;
		DOSBox_SetMenu();
	} else {
		menu.toggle=false;
		DOSBox_NoMenu();
	}
}
#endif

