/*
 *  Copyright (C) 2002-2011  The DOSBox Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/types.h>
#ifdef WIN32
#include <signal.h>
#include <process.h>
#endif

#include "cross.h"
#include "SDL.h"

#include "dosbox.h"
#include "video.h"
#include "mouse.h"
#include "pic.h"
#include "timer.h"
#include "setup.h"
#include "support.h"
#include "debug.h"
#include "render.h"
#include "menu.h"
#include "SDL_video.h"

#ifdef __WIN32__
#include "callback.h"
#include "dos_inc.h"
#include <malloc.h>
#include "Commdlg.h"
#include "windows.h"
#include <dirent.h>
#include "Shellapi.h"
#include "shell.h"
#include "SDL_syswm.h"
#include <cstring>
#include <fstream>
#include <sstream>
#endif // WIN32

#include "mapper.h"
#include "vga.h"
#include "keyboard.h"
#include "cpu.h"
#include "cross.h"
#include "control.h"
#include "glidedef.h"
#include "../save_state.h"

#define MAPPERFILE "mapper-" VERSION ".map"
//#define DISABLE_JOYSTICK

#if C_OPENGL
#include "SDL_opengl.h"

#ifndef APIENTRY
#define APIENTRY
#endif
#ifndef APIENTRYP
#define APIENTRYP APIENTRY *
#endif

#ifdef __WIN32__
#define NVIDIA_PixelDataRange 1

#ifndef WGL_NV_allocate_memory
#define WGL_NV_allocate_memory 1
typedef void * (APIENTRY * PFNWGLALLOCATEMEMORYNVPROC) (int size, float readfreq, float writefreq, float priority);
typedef void (APIENTRY * PFNWGLFREEMEMORYNVPROC) (void *pointer);
#endif

PFNWGLALLOCATEMEMORYNVPROC db_glAllocateMemoryNV = NULL;
PFNWGLFREEMEMORYNVPROC db_glFreeMemoryNV = NULL;

#else

#endif

extern void UI_Init();
extern void UI_Run(bool);

#if defined(NVIDIA_PixelDataRange)

#ifndef GL_NV_pixel_data_range
#define GL_NV_pixel_data_range 1
#define GL_WRITE_PIXEL_DATA_RANGE_NV      0x8878
typedef void (APIENTRYP PFNGLPIXELDATARANGENVPROC) (GLenum target, GLsizei length, GLvoid *pointer);
typedef void (APIENTRYP PFNGLFLUSHPIXELDATARANGENVPROC) (GLenum target);
#endif

PFNGLPIXELDATARANGENVPROC glPixelDataRangeNV = NULL;

#endif

#endif //C_OPENGL

#if !(ENVIRON_INCLUDED)
extern char** environ;
#endif

#ifdef WIN32
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#include <windows.h>
#if (HAVE_DDRAW_H)
#include <ddraw.h>
struct private_hwdata {
	LPDIRECTDRAWSURFACE3 dd_surface;
	LPDIRECTDRAWSURFACE3 dd_writebuf;
};
#endif

#if (HAVE_D3D9_H)
#include "direct3d.h"

CDirect3D* d3d = NULL;
#endif

#define STDOUT_FILE	TEXT("stdout.txt")
#define STDERR_FILE	TEXT("stderr.txt")
#define DEFAULT_CONFIG_FILE "/dosbox.conf"
#elif defined(MACOSX)
#define DEFAULT_CONFIG_FILE "/Library/Preferences/DOSBox Preferences"
#else /*linux freebsd*/
#define DEFAULT_CONFIG_FILE "/.dosboxrc"
#endif

#if C_SET_PRIORITY
#include <sys/resource.h>
#define PRIO_TOTAL (PRIO_MAX-PRIO_MIN)
#endif

#ifdef OS2
#define INCL_DOS
#define INCL_WIN
#include <os2.h>
#endif

enum SCREEN_TYPES	{
	SCREEN_OPENGLHQ,
	SCREEN_SURFACE,
	SCREEN_SURFACE_DDRAW,
	SCREEN_OVERLAY,
	SCREEN_OPENGL,
	SCREEN_DIRECT3D
};

enum PRIORITY_LEVELS {
	PRIORITY_LEVEL_PAUSE,
	PRIORITY_LEVEL_LOWEST,
	PRIORITY_LEVEL_LOWER,
	PRIORITY_LEVEL_NORMAL,
	PRIORITY_LEVEL_HIGHER,
	PRIORITY_LEVEL_HIGHEST
};

static bool load_videodrv=true;

struct SDL_Block {
	bool inited;
	bool active;							//If this isn't set don't draw
	bool updating;
	struct {
		Bit32u width;
		Bit32u height;
		Bit32u bpp;
		Bitu flags;
		double scalex,scaley;
		GFX_CallBack_t callback;
	} draw;
	bool wait_on_error;
	struct {
		struct {
			Bit16u width, height;
			bool fixed;
		} full;
		struct {
			Bit16u width, height;
		} window;
		Bit8u bpp;
		bool fullscreen;
		bool lazy_fullscreen;
		bool lazy_fullscreen_req;
		bool doublebuf;
		SCREEN_TYPES type;
		SCREEN_TYPES want_type;
	} desktop;
#if C_OPENGL
	struct {
		Bitu pitch;
		void * framebuf;
		GLuint texture;
		GLuint displaylist;
		GLint max_texsize;
		bool bilinear;
		bool packed_pixel;
		bool paletted_texture;
#if defined(NVIDIA_PixelDataRange)
		bool pixel_data_range;
#endif
	} opengl;
#endif
	struct {
		SDL_Surface * surface;
#if (HAVE_DDRAW_H) && defined(WIN32)
		RECT rect;
#endif
	} blit;
	struct {
		PRIORITY_LEVELS focus;
		PRIORITY_LEVELS nofocus;
	} priority;
	SDL_Rect clip;
	SDL_Surface * surface;
	SDL_Overlay * overlay;
	SDL_cond *cond;
	struct {
		bool autolock;
		bool autoenable;
		bool requestlock;
		bool locked;
		Bitu sensitivity;
	} mouse;
	SDL_Rect updateRects[1024];
	Bitu overscan_color;
	Bitu overscan_width;
	Bitu num_joysticks;
#if defined (WIN32)
	bool using_windib;
#endif
	// state of alt-keys for certain special handlings
	Bit8u laltstate;
	Bit8u raltstate;
};

static SDL_Block sdl;

extern const char* RunningProgram;
extern bool CPU_CycleAutoAdjust;
//Globals for keyboard initialisation
bool startup_state_numlock=false;
bool startup_state_capslock=false;

Bitu frames = 0;
#include "cpu.h"

void GFX_SetTitle(Bit32s cycles,Bits frameskip,Bits timing,bool paused){
	char title[200]={0};

	static Bit32s internal_cycles=0;
	static Bits internal_frameskip=0;
	static Bits internal_timing=0;
	if(cycles != -1) internal_cycles = cycles;
	if(frameskip != -1) internal_frameskip = frameskip;
	if(timing != -1) internal_timing = timing;

if(!menu_startup) { sprintf(title,"DOSBox %s, CPU speed: %8d cycles, Frameskip %2d, %8s",VERSION,internal_cycles,internal_frameskip,RunningProgram); SDL_WM_SetCaption(title,VERSION); return; }
if (menu.hidecycles) {
		if(CPU_CycleAutoAdjust) {
			sprintf(title,"DOSBox %s, CPU speed: max %3d%% cycles, Frameskip %2d, %8s",VERSION,CPU_CyclePercUsed,internal_frameskip,RunningProgram);
		} else {
			sprintf(title,"DOSBox %s, CPU speed: %8d cycles, Frameskip %2d, %8s",VERSION,internal_cycles,internal_frameskip,RunningProgram);
		}
	} else

	if(CPU_CycleAutoAdjust) {
		sprintf(title,"CPU : %s %8d%% = max %3d, %d FPS - %2d %8s %i.%i%%",core_mode,CPU_CyclePercUsed,internal_cycles,frames,internal_frameskip,RunningProgram,internal_timing/100,internal_timing%100/10);
	} else {
		sprintf(title,"CPU : %s %8d = %8d, %d FPS - %2d %8s %i.%i%%",core_mode,CPU_CyclesCur,internal_cycles,frames,internal_frameskip,RunningProgram,internal_timing/100,internal_timing%100/10);
	}

	if(paused) strcat(title," PAUSED");
	SDL_WM_SetCaption(title,VERSION);
}

static unsigned char logo[32*32*4]= {
#include "dosbox_logo.h"
};
static void DOSBox_SetOriginalIcon(void) {
#if !defined(MACOSX)
#if WORDS_BIGENDIAN
    	SDL_Surface* logos= SDL_CreateRGBSurfaceFrom((void*)logo,32,32,32,128,0xff000000,0x00ff0000,0x0000ff00,0);
#else
    	SDL_Surface* logos= SDL_CreateRGBSurfaceFrom((void*)logo,32,32,32,128,0x000000ff,0x0000ff00,0x00ff0000,0);
#endif
    	SDL_WM_SetIcon(logos,NULL);
#endif
}

void GFX_SetIcon(void) {
#if !defined(MACOSX)
	/* Set Icon (must be done before any sdl_setvideomode call) */
	/* But don't set it on OS X, as we use a nicer external icon there. */
	/* Made into a separate call, so it can be called again when we restart the graphics output on win32 */
    if(menu_compatible) { DOSBox_SetOriginalIcon(); return; }
#endif

#ifdef WIN32
    HICON hIcon1;
    hIcon1 = (HICON) LoadImage( GetModuleHandle(NULL), MAKEINTRESOURCE(dosbox_ico), IMAGE_ICON,
    16,
    16,
    LR_DEFAULTSIZE);
    SendMessage(GetHWND(), WM_SETICON, ICON_SMALL, (LPARAM) hIcon1 ); 
#endif
}

static void KillSwitch(bool pressed) {
	if (!pressed)
		return;
    if(sdl.desktop.fullscreen) GFX_SwitchFullScreen();
	throw 1;
}

static void SDL_Overscan(void) {
    sdl.overscan_color=0;
	if (sdl.overscan_width) {
		Bitu border_color =  GFX_GetRGB(vga.dac.rgb[vga.attr.overscan_color].red<<2,
			vga.dac.rgb[vga.attr.overscan_color].green<<2, vga.dac.rgb[vga.attr.overscan_color].blue<<2);
		if (border_color != sdl.overscan_color) {
			sdl.overscan_color = border_color;

		// Find four rectangles forming the border
			SDL_Rect *rect = &sdl.updateRects[0];
			rect->x = 0; rect->y = 0; rect->w = sdl.draw.width+2*sdl.clip.x; rect->h = sdl.clip.y; // top
			if (rect->h > sdl.overscan_width) { rect->y += (rect->h-sdl.overscan_width); rect->h = sdl.overscan_width; }
			if (sdl.clip.x > sdl.overscan_width) { rect->x += (sdl.clip.x-sdl.overscan_width); rect->w -= 2*(sdl.clip.x-sdl.overscan_width); }
			rect = &sdl.updateRects[1];
			rect->x = 0; rect->y = sdl.clip.y; rect->w = sdl.clip.x; rect->h = sdl.draw.height; // left
			if (rect->w > sdl.overscan_width) { rect->x += (rect->w-sdl.overscan_width); rect->w = sdl.overscan_width; }
			rect = &sdl.updateRects[2];
			rect->x = sdl.clip.x+sdl.draw.width; rect->y = sdl.clip.y; rect->w = sdl.clip.x; rect->h = sdl.draw.height; // right
			if (rect->w > sdl.overscan_width) { rect->w = sdl.overscan_width; }
			rect = &sdl.updateRects[3];
			rect->x = 0; rect->y = sdl.clip.y+sdl.draw.height; rect->w = sdl.draw.width+2*sdl.clip.x; rect->h = sdl.clip.y; // bottom
			if (rect->h > sdl.overscan_width) { rect->h = sdl.overscan_width; }
			if (sdl.clip.x > sdl.overscan_width) { rect->x += (sdl.clip.x-sdl.overscan_width); rect->w -= 2*(sdl.clip.x-sdl.overscan_width); }

			if (sdl.surface->format->BitsPerPixel == 8) { // SDL_FillRect seems to have some issues with palettized hw surfaces
				Bit8u* pixelptr = (Bit8u*)sdl.surface->pixels;
				Bitu linepitch = sdl.surface->pitch;
				for (Bits i=0; i<4; i++) {
					rect = &sdl.updateRects[i];
					Bit8u* start = pixelptr + rect->y*linepitch + rect->x;
					for (Bits j=0; j<rect->h; j++) {
						memset(start, vga.attr.overscan_color, rect->w);
						start += linepitch;
					}
				}
			} else {
				for (Bits i=0; i<4; i++)
				    SDL_FillRect(sdl.surface, &sdl.updateRects[i], border_color);
				SDL_UpdateRects(sdl.surface, 4, sdl.updateRects);
			}
		}
	}
}


static bool emu_paused;
bool DOSBox_Paused()
{
	return emu_paused;
}


static void PauseDOSBox(bool pressed) {
	if (!pressed)
		return;

#if 1
	GFX_SetTitle(-1,-1,-1,true);


	emu_paused = !emu_paused;

	if( emu_paused == 0 ) {
		// restore mouse state
		void GFX_UpdateSDLCaptureState();
		GFX_UpdateSDLCaptureState();
	}
	else {
		// give mouse to win32 (ex. alt-tab)
		SDL_WM_GrabInput(SDL_GRAB_OFF);
	}
	return;
#else
	GFX_SetTitle(-1,-1,-1,true);
	bool paused = true;
	KEYBOARD_ClrBuffer();
	SDL_Delay(500);
	SDL_Event event;
	while (SDL_PollEvent(&event)) {
		// flush event queue.
	}

	
	// give mouse to win32 (ex. alt-tab)
	SDL_WM_GrabInput(SDL_GRAB_OFF);


	while (paused) {
		SDL_WaitEvent(&event);    // since we're not polling, cpu usage drops to 0.
		if(event.type==SDL_SYSWMEVENT && event.syswm.msg->msg==WM_COMMAND && event.syswm.msg->wParam==ID_PAUSE) {
				paused=false;
				GFX_SetTitle(-1,-1,-1,false);	
				break;
		}
		switch (event.type) {

			case SDL_QUIT: KillSwitch(true); break;
			case SDL_KEYDOWN:   // Must use Pause/Break Key to resume.
			case SDL_KEYUP:
			if(event.key.keysym.sym == SDLK_PAUSE) {

				paused = false;
				GFX_SetTitle(-1,-1,-1,false);	
				break;
			}
#if defined (MACOSX)
			if (event.key.keysym.sym == SDLK_q && (event.key.keysym.mod == KMOD_RMETA || event.key.keysym.mod == KMOD_LMETA) ) {
				/* On macs, all aps exit when pressing cmd-q */
				KillSwitch(true);
				break;
			} 
#endif
		}
	}


	// restore mouse state
	void GFX_UpdateSDLCaptureState();
	GFX_UpdateSDLCaptureState();

	// redraw screen (ex. fullscreen - pause - alt+tab x2 - unpause)
	if (sdl.draw.callback) (sdl.draw.callback)( GFX_CallBackReset );
#endif
}

#if defined (WIN32)
bool GFX_SDLUsingWinDIB(void) {
	return sdl.using_windib;
}
#endif

static bool fullscreen_switch=true;
static void SDLScreen_Reset(void) {
	char* sdl_videodrv = getenv("SDL_VIDEODRIVER");
	if ((sdl_videodrv && !strcmp(sdl_videodrv,"windib")) || sdl.desktop.fullscreen || fullscreen_switch || sdl.desktop.want_type==SCREEN_OPENGLHQ || glide.enabled || menu_compatible) return;
    int id, major, minor;
    DOSBox_CheckOS(id, major, minor);
    if((id==VER_PLATFORM_WIN32_NT) && (major<6) || sdl.desktop.want_type==SCREEN_DIRECT3D) return;

	SDL_QuitSubSystem(SDL_INIT_VIDEO);	SDL_Delay(500);
	SDL_InitSubSystem(SDL_INIT_VIDEO);
	GFX_SetIcon();
	GFX_SetTitle(-1,-1,-1,false);
//	GFX_LosingFocus();
}

/* Reset the screen with current values in the sdl structure */
Bitu GFX_GetBestMode(Bitu flags) {
	Bitu testbpp,gotbpp;
	switch (sdl.desktop.want_type) {
	case SCREEN_OPENGLHQ:
		flags|=GFX_SCALING;
	case SCREEN_SURFACE:
check_surface:
		flags &= ~GFX_LOVE_8;		//Disable love for 8bpp modes
		/* Check if we can satisfy the depth it loves */
		if (flags & GFX_LOVE_8) testbpp=8;
		else if (flags & GFX_LOVE_15) testbpp=15;
		else if (flags & GFX_LOVE_16) testbpp=16;
		else if (flags & GFX_LOVE_32) testbpp=32;
		else testbpp=0;
#if (HAVE_DDRAW_H) && defined(WIN32)
check_gotbpp:
#endif
		if (sdl.desktop.fullscreen) gotbpp=SDL_VideoModeOK(640,480,testbpp,SDL_FULLSCREEN|SDL_HWSURFACE|SDL_HWPALETTE);
		else gotbpp=sdl.desktop.bpp;
		/* If we can't get our favorite mode check for another working one */
		switch (gotbpp) {
		case 8:
			if (flags & GFX_CAN_8) flags&=~(GFX_CAN_15|GFX_CAN_16|GFX_CAN_32);
			break;
		case 15:
			if (flags & GFX_CAN_15) flags&=~(GFX_CAN_8|GFX_CAN_16|GFX_CAN_32);
			break;
		case 16:
			if (flags & GFX_CAN_16) flags&=~(GFX_CAN_8|GFX_CAN_15|GFX_CAN_32);
			break;
		case 24:
		case 32:
			if (flags & GFX_CAN_32) flags&=~(GFX_CAN_8|GFX_CAN_15|GFX_CAN_16);
			break;
		}
		flags |= GFX_CAN_RANDOM;
		break;
#if (HAVE_DDRAW_H) && defined(WIN32)
	case SCREEN_SURFACE_DDRAW:
		if (!(flags&(GFX_CAN_15|GFX_CAN_16|GFX_CAN_32))) goto check_surface;
		if (flags & GFX_LOVE_15) testbpp=15;
		else if (flags & GFX_LOVE_16) testbpp=16;
		else if (flags & GFX_LOVE_32) testbpp=32;
		else testbpp=0;
		flags|=GFX_SCALING;
		goto check_gotbpp;
#endif
	case SCREEN_OVERLAY:
		if (flags & GFX_RGBONLY || !(flags&GFX_CAN_32)) goto check_surface;
		flags|=GFX_SCALING;
		flags&=~(GFX_CAN_8|GFX_CAN_15|GFX_CAN_16);
		break;
#if C_OPENGL
	case SCREEN_OPENGL:
		if (flags & GFX_RGBONLY || !(flags&GFX_CAN_32)) goto check_surface;
		flags|=GFX_SCALING;
		flags&=~(GFX_CAN_8|GFX_CAN_15|GFX_CAN_16);
		break;
#endif
#if (HAVE_D3D9_H) && defined(WIN32)
	case SCREEN_DIRECT3D:
		flags|=GFX_SCALING;
		if(GCC_UNLIKELY(d3d->bpp16))
		    flags&=~(GFX_CAN_8|GFX_CAN_15|GFX_CAN_32);
		else
		    flags&=~(GFX_CAN_8|GFX_CAN_15|GFX_CAN_16);
		break;
#endif
	default:
		goto check_surface;
		break;
	}
	return flags;
}

void SDL_Prepare(void) {
    if(menu_compatible) return;
    SDL_PumpEvents(); SDL_EventState(SDL_SYSWMEVENT, SDL_ENABLE);
    DragAcceptFiles(GetHWND(), TRUE);
}

void GFX_ResetScreen(void) {
	fullscreen_switch=false; 
	if(glide.enabled) {
		GLIDE_ResetScreen(true);
		return;
	}
	GFX_Stop();
	if (sdl.draw.callback)
		(sdl.draw.callback)( GFX_CallBackReset );
	GFX_Start();
	CPU_Reset_AutoAdjust();
    fullscreen_switch=true;
    if (!sdl.desktop.want_type==SCREEN_OPENGLHQ && !sdl.desktop.fullscreen && GetMenu(GetHWND()) == NULL)
    DOSBox_RefreshMenu(); // for menu
}

void GFX_ForceFullscreenExit(void) {
	if (sdl.desktop.lazy_fullscreen) {
//		sdl.desktop.lazy_fullscreen_req=true;
		LOG_MSG("GFX LF: invalid screen change");
	} else {
		sdl.desktop.fullscreen=false;
		GFX_ResetScreen();
	}
}

static int int_log2 (int val) {
    int log = 0;
    while ((val >>= 1) != 0)
	log++;
    return log;
}


static SDL_Surface * GFX_SetupSurfaceScaled(Bit32u sdl_flags, Bit32u bpp) {
	Bit16u fixedWidth;
	Bit16u fixedHeight;

	if (sdl.desktop.fullscreen) {
		fixedWidth = sdl.desktop.full.fixed ? sdl.desktop.full.width : 0;
		fixedHeight = sdl.desktop.full.fixed ? sdl.desktop.full.height : 0;
		sdl_flags |= SDL_FULLSCREEN|SDL_HWSURFACE;
	} else {
		fixedWidth = sdl.desktop.window.width;
		fixedHeight = sdl.desktop.window.height;
		sdl_flags |= SDL_HWSURFACE;
	}
	if (fixedWidth && fixedHeight) {
		double ratio_w=(double)fixedWidth/(sdl.draw.width*sdl.draw.scalex);
		double ratio_h=(double)fixedHeight/(sdl.draw.height*sdl.draw.scaley);
		if ( ratio_w < ratio_h) {
			sdl.clip.w=fixedWidth;
			sdl.clip.h=(Bit16u)(sdl.draw.height*sdl.draw.scaley*ratio_w);
		} else {
			sdl.clip.w=(Bit16u)(sdl.draw.width*sdl.draw.scalex*ratio_h);
			sdl.clip.h=(Bit16u)fixedHeight;
		}
		if (sdl.desktop.fullscreen)
			sdl.surface = SDL_SetVideoMode(fixedWidth,fixedHeight,bpp,sdl_flags);
		else
			sdl.surface = SDL_SetVideoMode(sdl.clip.w,sdl.clip.h,bpp,sdl_flags);
		if (sdl.surface && sdl.surface->flags & SDL_FULLSCREEN) {
			sdl.clip.x=(Sint16)((sdl.surface->w-sdl.clip.w)/2);
			sdl.clip.y=(Sint16)((sdl.surface->h-sdl.clip.h)/2);
		} else {
			sdl.clip.x = 0;
			sdl.clip.y = 0;
		}
		return sdl.surface;
	} else {
		sdl.clip.x=0;sdl.clip.y=0;
		sdl.clip.w=(Bit16u)(sdl.draw.width*sdl.draw.scalex);
		sdl.clip.h=(Bit16u)(sdl.draw.height*sdl.draw.scaley);
		sdl.surface=SDL_SetVideoMode(sdl.clip.w,sdl.clip.h,bpp,sdl_flags);
		return sdl.surface;
	}
}

void GFX_TearDown(void) {
	if (sdl.updating)
		GFX_EndUpdate( 0 );

	if (sdl.blit.surface) {
		SDL_FreeSurface(sdl.blit.surface);
		sdl.blit.surface=0;
	}
}

static void EndSplashScreen();

extern bool RENDER_GetAspect(void);
extern bool RENDER_GetAutofit(void);

static void GFX_ResetSDL() {
#ifdef WIN32
	if(!load_videodrv && !sdl.using_windib) {
		LOG_MSG("Resetting to WINDIB mode");
		SDL_QuitSubSystem(SDL_INIT_VIDEO);
		putenv("SDL_VIDEODRIVER=windib");
		sdl.using_windib=true;
		if (SDL_InitSubSystem(SDL_INIT_VIDEO)<0) E_Exit("Can't init SDL Video %s",SDL_GetError());
		GFX_SetIcon(); GFX_SetTitle(-1,-1,-1,false);
		if(!sdl.desktop.fullscreen && GetMenu(GetHWND()) == NULL) DOSBox_RefreshMenu();
	}
#endif
}

Bitu GFX_SetSize(Bitu width,Bitu height,Bitu flags,double scalex,double scaley,GFX_CallBack_t callback) {
	EndSplashScreen();
	if (sdl.updating)
		GFX_EndUpdate( 0 );

	sdl.draw.width=width;
	sdl.draw.height=height;
	sdl.draw.flags=flags;
	sdl.draw.callback=callback;
	sdl.draw.scalex=scalex;
	sdl.draw.scaley=scaley;

	Bitu bpp=0;
	Bitu retFlags = 0;

	if (sdl.blit.surface) {
		SDL_FreeSurface(sdl.blit.surface);
		sdl.blit.surface=0;
	}
	switch (sdl.desktop.want_type) {
	case SCREEN_OPENGLHQ:
		static char scale[64];
		if (flags & GFX_CAN_8) bpp=8;
		if (flags & GFX_CAN_15) bpp=15;
		if (flags & GFX_CAN_16) bpp=16;
		if (flags & GFX_CAN_32) bpp=32;
		sdl.desktop.type=SCREEN_SURFACE;
		sdl.clip.x=0;
		sdl.clip.y=0;
		if(!sdl.desktop.fullscreen) {
		    if(sdl.desktop.window.width && sdl.desktop.window.height) {
			scalex=(double)sdl.desktop.window.width/(sdl.draw.width*sdl.draw.scalex);
			scaley=(double)sdl.desktop.window.height/(sdl.draw.height*sdl.draw.scaley);
			if(scalex < scaley) {
			    sdl.clip.w=sdl.desktop.window.width;
			    sdl.clip.h=(Bit16u)(sdl.draw.height*sdl.draw.scaley*scalex);
			} else {
			    sdl.clip.w=(Bit16u)(sdl.draw.width*sdl.draw.scalex*scaley);
			    sdl.clip.h=(Bit16u)sdl.desktop.window.height;
			}
		    } else {
			sdl.clip.w=(Bit16u)(sdl.draw.width*sdl.draw.scalex);
			sdl.clip.h=(Bit16u)(sdl.draw.height*sdl.draw.scaley);
		    }
		    snprintf(scale,64,"SDL_OPENGLHQ_WINRES=%dx%d",sdl.clip.w,sdl.clip.h);
		    sdl.clip.w=width; sdl.clip.h=height;

    		} else if(!sdl.desktop.full.fixed) {
		    snprintf(scale,64,"SDL_OPENGLHQ_FULLRES=%dx%d",sdl.draw.width*(Uint16)sdl.draw.scalex,
								sdl.draw.height*(Uint16)sdl.draw.scaley);
		    sdl.clip.w=width; sdl.clip.h=height;
		} else {
		    snprintf(scale,64,"SDL_OPENGLHQ_FULLRES=%dx%d",sdl.desktop.full.width,sdl.desktop.full.height);
		    scalex=(double)sdl.desktop.full.width/(sdl.draw.width*sdl.draw.scalex);
		    scaley=(double)sdl.desktop.full.height/(sdl.draw.height*sdl.draw.scaley);
		    sdl.clip.w=width; sdl.clip.h=height;

		    if (scalex < scaley)
			height *= scaley/scalex;
		    else
			width *= scalex/scaley;
		    sdl.clip.x=(Sint16)((width-sdl.clip.w)/2);
		    sdl.clip.y=(Sint16)((height-sdl.clip.h)/2);
		}
		putenv(scale);
		sdl.surface=SDL_SetVideoMode(width,height,bpp,(sdl.desktop.fullscreen?SDL_FULLSCREEN:0)|SDL_HWSURFACE|SDL_ANYFORMAT);
		if (sdl.surface) {
		    switch (sdl.surface->format->BitsPerPixel) {
			case 8:retFlags = GFX_CAN_8;break;
			case 15:retFlags = GFX_CAN_15;break;
			case 16:retFlags = GFX_CAN_16;break;
			case 32:retFlags = GFX_CAN_32;break;
			default:break;
		    }
		    if (retFlags) {
			if (sdl.surface->flags & SDL_HWSURFACE)
			    retFlags |= GFX_HARDWARE;
			retFlags |= GFX_SCALING;
		    }
		}
		break;
	case SCREEN_SURFACE:
		GFX_ResetSDL();
dosurface:
		if (flags & GFX_CAN_8) bpp=8;
		if (flags & GFX_CAN_15) bpp=15;
		if (flags & GFX_CAN_16) bpp=16;
		if (flags & GFX_CAN_32) bpp=32;
		sdl.desktop.type=SCREEN_SURFACE;
		sdl.clip.w=width;
		sdl.clip.h=height;
		SDLScreen_Reset();
		if (sdl.desktop.fullscreen) {
			Uint32 flags = SDL_FULLSCREEN | SDL_HWPALETTE |
				((flags & GFX_CAN_RANDOM) ? SDL_SWSURFACE : SDL_HWSURFACE) |
				(sdl.desktop.doublebuf ? SDL_DOUBLEBUF|SDL_ASYNCBLIT : 0);
			if (sdl.desktop.full.fixed) {
				sdl.clip.x=(Sint16)((sdl.desktop.full.width-width)/2);
				sdl.clip.y=(Sint16)((sdl.desktop.full.height-height)/2);
				sdl.surface=SDL_SetVideoMode(sdl.desktop.full.width,
					sdl.desktop.full.height, bpp, flags);
			} else {
				sdl.clip.x=0; sdl.clip.y=0;
				sdl.surface=SDL_SetVideoMode(width, height, bpp, flags);
			}
			if (sdl.surface == NULL) {
				LOG_MSG("Fullscreen not supported: %s", SDL_GetError());
#ifdef WIN32
				if (sdl.using_windib) LOG_MSG("Please retry with DirectX driver.");
#endif
				sdl.desktop.fullscreen=false;
				GFX_CaptureMouse();
				goto dosurface;
			}
		} else {
			sdl.clip.x=sdl.overscan_width;sdl.clip.y=sdl.overscan_width;
			//Section_prop * section=static_cast<Section_prop *>(control->GetSection("sdl"));
			sdl.surface=SDL_SetVideoMode(width+2*sdl.overscan_width,height+2*sdl.overscan_width,bpp,(flags & GFX_CAN_RANDOM) ? SDL_SWSURFACE | SDL_RESIZABLE : SDL_HWSURFACE | SDL_RESIZABLE);
#ifdef WIN32
			if (sdl.surface == NULL) {
				SDL_QuitSubSystem(SDL_INIT_VIDEO);
				if (!sdl.using_windib) {
					LOG_MSG("Failed to create hardware surface.\nRestarting video subsystem with windib enabled.");
					putenv("SDL_VIDEODRIVER=windib");
					sdl.using_windib=true;
				} else {
					LOG_MSG("Failed to create hardware surface.\nRestarting video subsystem with directx enabled.");
					putenv("SDL_VIDEODRIVER=directx");
					sdl.using_windib=false;
				}
				SDL_InitSubSystem(SDL_INIT_VIDEO);
				GFX_SetIcon(); //Set Icon again
				sdl.surface = SDL_SetVideoMode(width,height,bpp,SDL_HWSURFACE);
				if(sdl.surface) GFX_SetTitle(-1,-1,-1,false); //refresh title.
			}
#endif
			if (sdl.surface == NULL)
				E_Exit("Could not set windowed video mode %ix%i-%i: %s",width,height,bpp,SDL_GetError());
		}
		if (sdl.surface) {
			switch (sdl.surface->format->BitsPerPixel) {
			case 8:
				retFlags = GFX_CAN_8;
                break;
			case 15:
				retFlags = GFX_CAN_15;
				break;
			case 16:
				retFlags = GFX_CAN_16;
                break;
			case 32:
				retFlags = GFX_CAN_32;
                break;
			}
			if (retFlags && (sdl.surface->flags & SDL_HWSURFACE))
				retFlags |= GFX_HARDWARE;
			if (retFlags && (sdl.surface->flags & SDL_DOUBLEBUF)) {
				sdl.blit.surface=SDL_CreateRGBSurface(SDL_HWSURFACE,
					sdl.draw.width, sdl.draw.height,
					sdl.surface->format->BitsPerPixel,
					sdl.surface->format->Rmask,
					sdl.surface->format->Gmask,
					sdl.surface->format->Bmask,
				0);
				/* If this one fails be ready for some flickering... */
			}
		}
		break;
#if (HAVE_DDRAW_H) && defined(WIN32)
	case SCREEN_SURFACE_DDRAW:
    {
		if(!load_videodrv && sdl.using_windib) {
			LOG_MSG("Resetting to DirectX mode");
			SDL_QuitSubSystem(SDL_INIT_VIDEO);
			putenv("SDL_VIDEODRIVER=directx");
			sdl.using_windib=false;
			if (SDL_InitSubSystem(SDL_INIT_VIDEO)<0) E_Exit("Can't init SDL Video %s",SDL_GetError());
			GFX_SetIcon(); GFX_SetTitle(-1,-1,-1,false);
			if(!sdl.desktop.fullscreen && GetMenu(GetHWND()) == NULL) DOSBox_RefreshMenu();
		}

		if (flags & GFX_CAN_15) bpp=15;
		if (flags & GFX_CAN_16) bpp=16;
		if (flags & GFX_CAN_32) bpp=32;
		SDLScreen_Reset();
        //Section_prop * section=static_cast<Section_prop *>(control->GetSection("sdl"));
		if (!GFX_SetupSurfaceScaled((sdl.desktop.doublebuf && sdl.desktop.fullscreen) ? SDL_DOUBLEBUF : SDL_RESIZABLE,bpp)) goto dosurface;

		
		sdl.blit.rect.top=sdl.clip.y+sdl.overscan_width;
		sdl.blit.rect.left=sdl.clip.x+sdl.overscan_width;
		sdl.blit.rect.right=sdl.clip.x+sdl.clip.w;
		sdl.blit.rect.bottom=sdl.clip.y+sdl.clip.h;
		sdl.blit.surface=SDL_CreateRGBSurface(SDL_HWSURFACE,sdl.draw.width+sdl.overscan_width*2,sdl.draw.height+sdl.overscan_width*2,
				sdl.surface->format->BitsPerPixel,
				sdl.surface->format->Rmask,
				sdl.surface->format->Gmask,
				sdl.surface->format->Bmask,
				0);
		if (!sdl.blit.surface || (!sdl.blit.surface->flags&SDL_HWSURFACE)) {
			if (sdl.blit.surface) {
				SDL_FreeSurface(sdl.blit.surface);
				sdl.blit.surface=0;
			}
			sdl.desktop.want_type=SCREEN_SURFACE;
			LOG_MSG("Failed to create ddraw surface, back to normal surface.");
			goto dosurface;
		}
		switch (sdl.surface->format->BitsPerPixel) {
		case 15:
			retFlags = GFX_CAN_15 | GFX_SCALING | GFX_HARDWARE;
			break;
		case 16:
			retFlags = GFX_CAN_16 | GFX_SCALING | GFX_HARDWARE;
               break;
		case 32:
			retFlags = GFX_CAN_32 | GFX_SCALING | GFX_HARDWARE;
               break;
		}
		sdl.desktop.type=SCREEN_SURFACE_DDRAW;
		break;
    }
#endif
	case SCREEN_OVERLAY:
    {
	    GFX_ResetSDL();
		if (sdl.overlay) {
			SDL_FreeYUVOverlay(sdl.overlay);
			sdl.overlay=0;
		}
		if (!(flags&GFX_CAN_32) || (flags & GFX_RGBONLY)) goto dosurface;
		SDLScreen_Reset();
		//Section_prop * sec=static_cast<Section_prop *>(control->GetSection("sdl"));
		if (!GFX_SetupSurfaceScaled(SDL_RESIZABLE,0)) goto dosurface;
		sdl.overlay=SDL_CreateYUVOverlay(width*2,height,SDL_UYVY_OVERLAY,sdl.surface);
		if (!sdl.overlay) {
			LOG_MSG("SDL:Failed to create overlay, switching back to surface");
			goto dosurface;
		}
		sdl.desktop.type=SCREEN_OVERLAY;
		retFlags = GFX_CAN_32 | GFX_SCALING | GFX_HARDWARE;
		break;
    }
#if C_OPENGL
	case SCREEN_OPENGL:
	{
	    GFX_ResetSDL();
		if (sdl.opengl.framebuf) {
#if defined(NVIDIA_PixelDataRange)
			if (sdl.opengl.pixel_data_range) db_glFreeMemoryNV(sdl.opengl.framebuf);
			else
#endif
			free(sdl.opengl.framebuf);
		}
		sdl.opengl.framebuf=0;
		if (!(flags&GFX_CAN_32) || (flags & GFX_RGBONLY)) goto dosurface;
		// SDLScreen_Reset();
		int texsize=2 << int_log2(width > height ? width : height);
		if (texsize>sdl.opengl.max_texsize) {
			LOG_MSG("SDL:OPENGL:No support for texturesize of %d, falling back to surface",texsize);
			goto dosurface;
		}
		SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
#if SDL_VERSION_ATLEAST(1, 2, 11)
//	if(menu.startup) {
		Section_prop * sec=static_cast<Section_prop *>(control->GetSection("vsync"));
		if(sec) {
			if(!strcmp(sec->Get_string("vsyncmode"),"host")) 
        		SDL_GL_SetAttribute( SDL_GL_SWAP_CONTROL, 1 );
            else
           		SDL_GL_SetAttribute( SDL_GL_SWAP_CONTROL, 0 );
		}
//	}
#endif
		//Section_prop * sec=static_cast<Section_prop *>(control->GetSection("sdl"));
		GFX_SetupSurfaceScaled(SDL_OPENGL|SDL_RESIZABLE,0);
		if (!sdl.surface || sdl.surface->format->BitsPerPixel<15) {
			LOG_MSG("SDL:OPENGL:Can't open drawing surface, are you running in 16bpp(or higher) mode?");
			goto dosurface;
		}
		/* Create the texture and display list */
#if defined(NVIDIA_PixelDataRange)
		if (sdl.opengl.pixel_data_range) {
			sdl.opengl.framebuf=db_glAllocateMemoryNV(width*height*4,0.0,1.0,1.0);
			glPixelDataRangeNV(GL_WRITE_PIXEL_DATA_RANGE_NV,width*height*4,sdl.opengl.framebuf);
			glEnableClientState(GL_WRITE_PIXEL_DATA_RANGE_NV);
		} else {
#else
		{
#endif
			sdl.opengl.framebuf=malloc(width*height*4);		//32 bit color
		}
		sdl.opengl.pitch=width*4;
		glViewport(sdl.clip.x,sdl.clip.y,sdl.clip.w,sdl.clip.h);
		glMatrixMode (GL_PROJECTION);
		glDeleteTextures(1,&sdl.opengl.texture);
 		glGenTextures(1,&sdl.opengl.texture);
		glBindTexture(GL_TEXTURE_2D,sdl.opengl.texture);
		// No borders
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		if (sdl.opengl.bilinear) {
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		} else {
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		}

		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, texsize, texsize, 0, GL_BGRA_EXT, GL_UNSIGNED_BYTE, 0);

		glClearColor (0.0, 0.0, 0.0, 1.0);
		glClear(GL_COLOR_BUFFER_BIT);
		SDL_GL_SwapBuffers();
		glClear(GL_COLOR_BUFFER_BIT);
		glShadeModel (GL_FLAT);
		glDisable (GL_DEPTH_TEST);
		glDisable (GL_LIGHTING);
		glDisable(GL_CULL_FACE);
		glEnable(GL_TEXTURE_2D);
		glMatrixMode (GL_MODELVIEW);
		glLoadIdentity ();

		GLfloat tex_width=((GLfloat)(width)/(GLfloat)texsize);
		GLfloat tex_height=((GLfloat)(height)/(GLfloat)texsize);

		if (glIsList(sdl.opengl.displaylist)) glDeleteLists(sdl.opengl.displaylist, 1);
		sdl.opengl.displaylist = glGenLists(1);
		glNewList(sdl.opengl.displaylist, GL_COMPILE);
		glBindTexture(GL_TEXTURE_2D, sdl.opengl.texture);
		glBegin(GL_QUADS);
		// lower left
		glTexCoord2f(0,tex_height); glVertex2f(-1.0f,-1.0f);
		// lower right
		glTexCoord2f(tex_width,tex_height); glVertex2f(1.0f, -1.0f);
		// upper right
		glTexCoord2f(tex_width,0); glVertex2f(1.0f, 1.0f);
		// upper left
		glTexCoord2f(0,0); glVertex2f(-1.0f, 1.0f);
		glEnd();
		glEndList();
		sdl.desktop.type=SCREEN_OPENGL;
		retFlags = GFX_CAN_32 | GFX_SCALING;
#if defined(NVIDIA_PixelDataRange)
		if (sdl.opengl.pixel_data_range)
			retFlags |= GFX_HARDWARE;
#endif
	break;
		}//OPENGL
#endif	//C_OPENGL
#if (HAVE_D3D9_H) && defined(WIN32)
	    case SCREEN_DIRECT3D: {
		// Calculate texture size
		if((!d3d->square) && (!d3d->pow2)) {
		    d3d->dwTexWidth=width;
		    d3d->dwTexHeight=height;
		} else if(d3d->square) {
		    int texsize=2 << int_log2(width > height ? width : height);
		    d3d->dwTexWidth=d3d->dwTexHeight=texsize;
		} else {
		    d3d->dwTexWidth=2 << int_log2(width);
		    d3d->dwTexHeight=2 << int_log2(height);
		}

		sdl.clip.x=0; sdl.clip.y=0;
		if(sdl.desktop.fullscreen) {
		    if(sdl.desktop.full.fixed) {
			sdl.clip.w=sdl.desktop.full.width;
			sdl.clip.h=sdl.desktop.full.height;
			scalex=(double)sdl.desktop.full.width/width;
			scaley=(double)sdl.desktop.full.height/height;
		    }
		} else {
		    if((sdl.desktop.window.width) && (sdl.desktop.window.height)) {
			scalex=(double)sdl.desktop.window.width/(sdl.draw.width*sdl.draw.scalex);
			scaley=(double)sdl.desktop.window.height/(sdl.draw.height*sdl.draw.scaley);
			if(scalex < scaley) {
			    sdl.clip.w=sdl.desktop.window.width;
			    sdl.clip.h=(Bit16u)(sdl.draw.height*sdl.draw.scaley*scalex);
			} else {
			    sdl.clip.w=(Bit16u)(sdl.draw.width*sdl.draw.scalex*scaley);
			    sdl.clip.h=(Bit16u)sdl.desktop.window.height;
			}
			scalex=(double)sdl.clip.w/width;
			scaley=(double)sdl.clip.h/height;
		    } else {
			sdl.clip.w=(Bit16u)(width*scalex);
			sdl.clip.h=(Bit16u)(height*scaley);
		    }
		}

		Section_prop *section=static_cast<Section_prop *>(control->GetSection("sdl"));
		if(section) {
		    d3d->LoadPixelShader(section->Get_string("pixelshader"), scalex, scaley);
		} else {
		    LOG_MSG("SDL:D3D:Could not get pixelshader info, shader disabled");
		}

		d3d->aspect=RENDER_GetAspect();
		d3d->autofit=RENDER_GetAutofit();
		if((sdl.desktop.fullscreen) && (!sdl.desktop.full.fixed)) {
		    // Don't do aspect ratio correction when fullfixed=false + aspect=false
			if(d3d->aspect == 0)
		    d3d->aspect=2;
	    sdl.clip.w=(Uint16)scalex;
	    sdl.clip.h=(Uint16)scaley;
	    // Do fullscreen scaling if pixel shaders are enabled
	    // or the game uses some weird resolution
	    if((d3d->psActive) || (sdl.clip.w != sdl.clip.h)) {
				sdl.clip.w*=width;
				sdl.clip.h*=height;
			} else { // just use native resolution
				sdl.clip.w=width;
				sdl.clip.h=height;
			}
		} else if(!sdl.desktop.fullscreen) d3d->aspect=-1;

		// Create a dummy sdl surface
		// D3D will hang or crash when using fullscreen with ddraw surface, therefore we hack SDL to provide
		// a GDI window with an additional 0x40 flag. If this fails or stock SDL is used, use WINDIB output
		if(GCC_UNLIKELY(d3d->bpp16)) {
		    sdl.surface=SDL_SetVideoMode(sdl.clip.w,sdl.clip.h,16,sdl.desktop.fullscreen ? SDL_FULLSCREEN|0x40 : SDL_RESIZABLE|0x40);
		    retFlags = GFX_CAN_16 | GFX_SCALING;
		} else {
		    sdl.surface=SDL_SetVideoMode(sdl.clip.w,sdl.clip.h,0,sdl.desktop.fullscreen ? SDL_FULLSCREEN|0x40 : SDL_RESIZABLE|0x40);
		    retFlags = GFX_CAN_32 | GFX_SCALING;
		}

		if (sdl.surface == NULL) E_Exit("Could not set video mode %ix%i-%i: %s",sdl.clip.w,sdl.clip.h,
					d3d->bpp16 ? 16:32,SDL_GetError());
		sdl.desktop.type=SCREEN_DIRECT3D;

		if(d3d->dynamic) retFlags |= GFX_HARDWARE;

		if(GCC_UNLIKELY(d3d->Resize3DEnvironment(sdl.clip.w,sdl.clip.h,width,
						    height,sdl.desktop.fullscreen) != S_OK)) {
		    retFlags = 0;
		}
#if LOG_D3D
		LOG_MSG("SDL:D3D:Display mode set to: %dx%d with %fx%f scale",
				    sdl.clip.w, sdl.clip.h,sdl.draw.scalex, sdl.draw.scaley);
#endif
		break;
	    }
#endif
	default:
		goto dosurface;
		break;
	}//CASE
	if (retFlags)
		GFX_Start();
	if (!sdl.mouse.autoenable) SDL_ShowCursor(sdl.mouse.autolock?SDL_DISABLE:SDL_ENABLE);
	return retFlags;
}

void GFX_CaptureMouse(void) {
	sdl.mouse.locked=!sdl.mouse.locked;
	if (sdl.mouse.locked) {
		SDL_WM_GrabInput(SDL_GRAB_ON);
		SDL_ShowCursor(SDL_DISABLE);
	} else {
		SDL_WM_GrabInput(SDL_GRAB_OFF);
		if (sdl.mouse.autoenable || !sdl.mouse.autolock) SDL_ShowCursor(SDL_ENABLE);
	}
        mouselocked=sdl.mouse.locked;
}

void GFX_UpdateSDLCaptureState(void) {
	if (sdl.mouse.locked) {
		SDL_WM_GrabInput(SDL_GRAB_ON);
		SDL_ShowCursor(SDL_DISABLE);
	} else {
		SDL_WM_GrabInput(SDL_GRAB_OFF);
		if (sdl.mouse.autoenable || !sdl.mouse.autolock) SDL_ShowCursor(SDL_ENABLE);
	}
	CPU_Reset_AutoAdjust();
	GFX_SetTitle(-1,-1,-1,false);
}

bool mouselocked; //Global variable for mapper
static void CaptureMouse(bool pressed) {
	if (!pressed)
		return;
	GFX_CaptureMouse();
}

#if defined (WIN32)
STICKYKEYS stick_keys = {sizeof(STICKYKEYS), 0};
void sticky_keys(bool restore){
	static bool inited = false;
	if (!inited){
		inited = true;
		SystemParametersInfo(SPI_GETSTICKYKEYS, sizeof(STICKYKEYS), &stick_keys, 0);
	} 
	if (restore) {
		SystemParametersInfo(SPI_SETSTICKYKEYS, sizeof(STICKYKEYS), &stick_keys, 0);
		return;
	}
	//Get current sticky keys layout:
	STICKYKEYS s = {sizeof(STICKYKEYS), 0};
	SystemParametersInfo(SPI_GETSTICKYKEYS, sizeof(STICKYKEYS), &s, 0);
	if ( !(s.dwFlags & SKF_STICKYKEYSON)) { //Not on already
		s.dwFlags &= ~SKF_HOTKEYACTIVE;
		SystemParametersInfo(SPI_SETSTICKYKEYS, sizeof(STICKYKEYS), &s, 0);
	}
}
#else
#define sticky_keys(a)
#endif

static void opengl_init(void) {
	sdl.surface=SDL_SetVideoMode(640,400,0,SDL_OPENGL);
//	if (sdl.surface == NULL) {
//		LOG_MSG("Could not initialize OpenGL, switching back to surface");
//		sdl.desktop.want_type=SCREEN_SURFACE;
//	} else {
	sdl.opengl.framebuf=0;
	sdl.opengl.texture=0;
	sdl.opengl.displaylist=0;
	glGetIntegerv (GL_MAX_TEXTURE_SIZE, &sdl.opengl.max_texsize);
#if defined(__WIN32__) && defined(NVIDIA_PixelDataRange)
	glPixelDataRangeNV = (PFNGLPIXELDATARANGENVPROC) wglGetProcAddress("glPixelDataRangeNV");
	db_glAllocateMemoryNV = (PFNWGLALLOCATEMEMORYNVPROC) wglGetProcAddress("wglAllocateMemoryNV");
	db_glFreeMemoryNV = (PFNWGLFREEMEMORYNVPROC) wglGetProcAddress("wglFreeMemoryNV");
#endif
	const char * gl_ext = (const char *)glGetString (GL_EXTENSIONS);
	if(gl_ext && *gl_ext){
		sdl.opengl.packed_pixel=(strstr(gl_ext,"EXT_packed_pixels") > 0);
		sdl.opengl.paletted_texture=(strstr(gl_ext,"EXT_paletted_texture") > 0);
#if defined(NVIDIA_PixelDataRange)
		sdl.opengl.pixel_data_range=(strstr(gl_ext,"GL_NV_pixel_data_range") >0 ) &&
			glPixelDataRangeNV && db_glAllocateMemoryNV && db_glFreeMemoryNV;
		sdl.opengl.pixel_data_range = 0;
#endif
    	} else {
		sdl.opengl.packed_pixel=sdl.opengl.paletted_texture=false;
	}
}

#ifdef __WIN32__
static void d3d_init(void) {
	void change_output(int output);
	change_output(2);
	sdl.desktop.want_type=SCREEN_DIRECT3D;
	if(!load_videodrv && !sdl.using_windib) {
		LOG_MSG("Resetting to WINDIB mode");
		SDL_QuitSubSystem(SDL_INIT_VIDEO);
		putenv("SDL_VIDEODRIVER=windib");
		sdl.using_windib=true;
		if (SDL_InitSubSystem(SDL_INIT_VIDEO)<0) E_Exit("Can't init SDL Video %s",SDL_GetError());
		GFX_SetIcon(); GFX_SetTitle(-1,-1,-1,false);
		if(!sdl.desktop.fullscreen && GetMenu(GetHWND()) == NULL) DOSBox_RefreshMenu();
	}
	SDL_SysWMinfo wmi;
	SDL_VERSION(&wmi.version);

	if(!SDL_GetWMInfo(&wmi)) {
		LOG_MSG("SDL:Error retrieving window information");
		E_Exit("Failed to get window info");
	}

	if(sdl.desktop.fullscreen) {
		GFX_CaptureMouse();
	}

	    if(d3d) delete d3d;
	    d3d = new CDirect3D(640,400);

	    if(!d3d) E_Exit("Failed to create d3d object");

	    if(d3d->InitializeDX(wmi.window,sdl.desktop.doublebuf) != S_OK)
		    E_Exit("Unable to initialize DirectX");
}
#endif

static void openglhq_init(void) {
#ifdef WIN32
    DOSBox_NoMenu(); menu.gui=false;
    HMENU m_handle=GetMenu(GetHWND());
    if(m_handle) RemoveMenu(m_handle,0,0);
    DestroyWindow(GetHWND());
#endif
    sdl.overlay=0;
		char *oldvideo = getenv("SDL_VIDEODRIVER");

		if (oldvideo && strcmp(oldvideo,"openglhq")) {
		    char *driver = (char *)malloc(strlen(oldvideo)+strlen("SDL_OPENGLHQ_VIDEODRIVER=")+1);
		    strcpy(driver,"SDL_OPENGLHQ_VIDEODRIVER=");
		    strcat(driver,oldvideo);
		    putenv(driver);
		    free(driver);
		}
		if (sdl.desktop.doublebuf) putenv("SDL_OPENGLHQ_DOUBLEBUF=1");
		SDL_QuitSubSystem(SDL_INIT_VIDEO);
		putenv("SDL_VIDEODRIVER=openglhq");
		SDL_InitSubSystem(SDL_INIT_VIDEO);
		DOSBox_SetOriginalIcon();
		if(!menu_compatible) { SDL_PumpEvents(); SDL_EventState(SDL_SYSWMEVENT, SDL_ENABLE); }
		GFX_SetTitle(-1,-1,-1,false);
		sdl.desktop.want_type=SCREEN_OPENGLHQ;
}

void res_init(void) {
	if(sdl.desktop.want_type==SCREEN_OPENGLHQ) return;
	Section * sec = control->GetSection("sdl");
	Section_prop * section=static_cast<Section_prop *>(sec);
	sdl.desktop.full.fixed=false;
	const char* fullresolution=section->Get_string("fullresolution");
	sdl.desktop.full.width  = 0; sdl.desktop.full.height = 0;
	if(fullresolution && *fullresolution) {
		char res[100];
		safe_strncpy( res, fullresolution, sizeof( res ));
		fullresolution = lowcase (res);//so x and X are allowed
		if(strcmp(fullresolution,"original")) {
			sdl.desktop.full.fixed = true;
			char* height = const_cast<char*>(strchr(fullresolution,'x'));
			if(height && * height) {
				*height = 0;
				sdl.desktop.full.height = atoi(height+1);
				sdl.desktop.full.width  = atoi(res);
			}
		}
	}

	sdl.desktop.window.width  = 0;
	sdl.desktop.window.height = 0;
	const char* windowresolution=section->Get_string("windowresolution");
	if(windowresolution && *windowresolution) {
		//if(sdl.desktop.type==SCREEN_SURFACE) return;
		char res[100];
		safe_strncpy( res,windowresolution, sizeof( res ));
		windowresolution = lowcase (res);//so x and X are allowed
		if(strcmp(windowresolution,"original")) {
			char* height = const_cast<char*>(strchr(windowresolution,'x'));
			if(height && *height) {
				*height = 0;
				sdl.desktop.window.height = (Bit16u)atoi(height+1);
				sdl.desktop.window.width  = (Bit16u)atoi(res);
			}
		}
	}
	sdl.desktop.doublebuf=section->Get_bool("fulldouble");

	int width, height;
	SDL_GetDesktopMode(&width, &height);
	if (!sdl.desktop.full.width) {
		sdl.desktop.full.width = width;
	}
	if (!sdl.desktop.full.height) {
		sdl.desktop.full.height = height;
	}
	if(sdl.desktop.type==SCREEN_SURFACE && !sdl.desktop.fullscreen) return;
	else {
        if (glide.enabled) {
            DOSBox_RefreshMenu();
            GLIDE_ResetScreen(1);
        }
        else {
        	GFX_Stop();
        	if (sdl.draw.callback)
        		(sdl.draw.callback)( GFX_CallBackReset );
        	GFX_Start();
        }
    }
}

void res_input(bool type, const char * res) {
	Section* sec = control->GetSection("sdl");
	char win_res[11];
	if(sec) {
		strcpy(win_res,res);
		sec->ExecuteDestroy(false);
		if(type) {
			std::string tmp("windowresolution="); tmp.append(win_res);
			sec->HandleInputline(tmp);
		} else {
			std::string tmp("fullresolution="); tmp.append(win_res);
			sec->HandleInputline(tmp); }
		sec->ExecuteInit(false);

		res_init();
	}
}

void change_output(int output) {
	GFX_Stop();
	Section * sec = control->GetSection("sdl");
	Section_prop * section=static_cast<Section_prop *>(sec);
	sdl.overscan_width=section->Get_int("overscan");
	switch (output) {
	case 0:
		sdl.desktop.want_type=SCREEN_SURFACE;
//        sdl.overlay=0;
		break;
	case 1:
#ifdef WIN32
 	    sdl.surface=SDL_SetVideoMode(640,400,0,SDL_HWSURFACE|SDL_HWPALETTE);
		sdl.desktop.want_type=SCREEN_SURFACE_DDRAW;
#else
		sdl.desktop.want_type=SCREEN_SURFACE;
#endif
//        sdl.overlay=0;
		break;
	case 2:
		sdl.desktop.want_type=SCREEN_OVERLAY;
//        sdl.overlay=0;
		break;
	case 3:
		change_output(2);
		sdl.desktop.want_type=SCREEN_OPENGL;
		sdl.opengl.bilinear=true;
		opengl_init();
		break;
	case 4:
		change_output(2);
		sdl.desktop.want_type=SCREEN_OPENGL;
		sdl.opengl.bilinear=false;
		opengl_init();
		break;
#ifdef __WIN32__
		case 5:
		sdl.desktop.want_type=SCREEN_DIRECT3D;
		d3d_init();
		break;
#endif
		case 6: {
#ifdef __WIN32__
		char warn[100];
		sprintf(warn,"GUI will be disabled if output is set to OpenglHQ. Do you want to continue?");
		if (MessageBox(GetHWND(),warn,"Warning",MB_YESNO)==IDNO) { GFX_Stop(); GFX_Start(); return; }
#endif
		openglhq_init();
		}
		break;
	case 7:
		// do not set want_type
		break;
	case 8:
		if(sdl.desktop.want_type==SCREEN_OPENGL) opengl_init();
#ifdef WIN32
		else if(sdl.desktop.want_type==SCREEN_DIRECT3D) { if(sdl.desktop.fullscreen) GFX_CaptureMouse(); d3d_init(); }
#endif
		break;
	default:
		LOG_MSG("SDL:Unsupported output device %d, switching back to surface",output);
		sdl.desktop.want_type=SCREEN_SURFACE;
		break;
	}
	const char* windowresolution=section->Get_string("windowresolution");
	if(windowresolution && *windowresolution) {
		char res[100];
		safe_strncpy( res,windowresolution, sizeof( res ));
		windowresolution = lowcase (res);//so x and X are allowed
		if(strcmp(windowresolution,"original")) {
			if(output == 0) {
				std::string tmp("windowresolution=original");
				sec->HandleInputline(tmp);
				sec->ExecuteInit(false);
			}
		}
	}
	res_init();

	if (glide.enabled)
		GLIDE_ResetScreen();
	else {
       	if (sdl.draw.callback)
		(sdl.draw.callback)( GFX_CallBackReset );
		if(sdl.desktop.want_type==SCREEN_OPENGLHQ) {
			SetVal("render","scaler",!render.scale.forced?"hardware2x":"hardware2x forced");
			if(!menu.compatible) {
				SDL_PumpEvents();
				SDL_EventState(SDL_SYSWMEVENT, SDL_ENABLE);
			}
			static SDL_Surface* screen_surf;
			Bit32u rmask = 0x000000ff;
			Bit32u gmask = 0x0000ff00;
			Bit32u bmask = 0x00ff0000;
			screen_surf = SDL_CreateRGBSurface(SDL_SWSURFACE, 640, 400, 32, rmask, gmask, bmask, 0);
			Bit32u lasttick=GetTicks();
			for(Bitu i = 0; i <=5; i++) {
				if((GetTicks()-lasttick)>20) i++;
				while((GetTicks()-lasttick)<15) SDL_Delay(5);
				lasttick = GetTicks();
				SDL_SetAlpha(screen_surf, SDL_SRCALPHA,(Bit8u)(51*i));
				SDL_BlitSurface(screen_surf, NULL, sdl.surface, NULL);
				SDL_Flip(sdl.surface);
			}
			SDL_FreeSurface(screen_surf);
		}
    }
	GFX_SetTitle(CPU_CycleMax,-1,-1,false);
}

void GFX_SwitchFullScreen(void) {
    menu.resizeusing=true;
	sdl.desktop.fullscreen=!sdl.desktop.fullscreen;
	if (sdl.desktop.fullscreen) {
        if(sdl.desktop.want_type != SCREEN_OPENGLHQ) { if(!glide.enabled && menu.gui) SetMenu(GetHWND(),NULL); }
		if (!sdl.mouse.locked) GFX_CaptureMouse();
#if defined (WIN32)
		sticky_keys(false); //disable sticky keys in fullscreen mode
#endif
	} else {
		if (sdl.mouse.locked) GFX_CaptureMouse();
#if defined (WIN32)		
		sticky_keys(true); //restore sticky keys to default state in windowed mode.
#endif
	}
	if (glide.enabled)
		GLIDE_ResetScreen();
	else
		GFX_ResetScreen();
#ifdef WIN32
	if(menu.startup) {
		Section_prop * sec=static_cast<Section_prop *>(control->GetSection("vsync"));
		if(sec) {
			if(!strcmp(sec->Get_string("vsyncmode"),"host")) SetVal("vsync","vsyncmode","host");
		}
	}
#endif //WIN32
}

static void SwitchFullScreen(bool pressed) {
	if (!pressed)
		return;

	GFX_LosingFocus();
	if (sdl.desktop.lazy_fullscreen) {
		LOG_MSG("GFX LF: fullscreen switching not supported");
	} else {
		GFX_SwitchFullScreen();
	}
}

void GFX_SwitchLazyFullscreen(bool lazy) {
	sdl.desktop.lazy_fullscreen=lazy;
	sdl.desktop.lazy_fullscreen_req=false;
}

void GFX_SwitchFullscreenNoReset(void) {
	sdl.desktop.fullscreen=!sdl.desktop.fullscreen;
}

bool GFX_LazyFullscreenRequested(void) {
	if (sdl.desktop.lazy_fullscreen) return sdl.desktop.lazy_fullscreen_req;
	return false;
}

void GFX_RestoreMode(void) {
	if (!sdl.draw.callback) return;
	GFX_SetSize(sdl.draw.width,sdl.draw.height,sdl.draw.flags,sdl.draw.scalex,sdl.draw.scaley,sdl.draw.callback);
	GFX_UpdateSDLCaptureState();
}


bool GFX_StartUpdate(Bit8u * & pixels,Bitu & pitch) {
	if (!sdl.active || sdl.updating)
		return false;
	switch (sdl.desktop.type) {
	case SCREEN_SURFACE:
		if (sdl.blit.surface) {
			if (SDL_MUSTLOCK(sdl.blit.surface) && SDL_LockSurface(sdl.blit.surface))
				return false;
			pixels=(Bit8u *)sdl.blit.surface->pixels;
			pitch=sdl.blit.surface->pitch;
		} else {
			if (SDL_MUSTLOCK(sdl.surface) && SDL_LockSurface(sdl.surface))
				return false;
			pixels=(Bit8u *)sdl.surface->pixels;
			pixels+=sdl.clip.y*sdl.surface->pitch;
			pixels+=sdl.clip.x*sdl.surface->format->BytesPerPixel;
			pitch=sdl.surface->pitch;
		}
        SDL_Overscan();
		sdl.updating=true;
		return true;
#if (HAVE_DDRAW_H) && defined(WIN32)
	case SCREEN_SURFACE_DDRAW:
		if (SDL_LockSurface(sdl.blit.surface)) {
//			LOG_MSG("SDL Lock failed");
			return false;
		}
		pixels=(Bit8u *)sdl.blit.surface->pixels;
		pitch=sdl.blit.surface->pitch;
        SDL_Overscan();
		sdl.updating=true;
		return true;
#endif
	case SCREEN_OVERLAY:
		if (SDL_LockYUVOverlay(sdl.overlay)) return false;
		pixels=(Bit8u *)*(sdl.overlay->pixels);
		pitch=*(sdl.overlay->pitches);
		sdl.updating=true;
		return true;
#if C_OPENGL
	case SCREEN_OPENGL:
		pixels=(Bit8u *)sdl.opengl.framebuf;
		pitch=sdl.opengl.pitch;
		sdl.updating=true;
		return true;
#endif
#if (HAVE_D3D9_H) && defined(WIN32)
	case SCREEN_DIRECT3D:
		sdl.updating=d3d->LockTexture(pixels, pitch);
		return sdl.updating;
#endif
	default:
		break;
	}
	return false;
}


void GFX_EndUpdate( const Bit16u *changedLines ) {
#if (HAVE_DDRAW_H) && defined(WIN32)
	int ret;
#endif
	if (!sdl.updating)
		return;
	sdl.updating=false;
	switch (sdl.desktop.type) {
	case SCREEN_SURFACE:
		if (SDL_MUSTLOCK(sdl.surface)) {
			if (sdl.blit.surface) {
				SDL_UnlockSurface(sdl.blit.surface);
				int Blit = SDL_BlitSurface( sdl.blit.surface, 0, sdl.surface, &sdl.clip );
				LOG(LOG_MISC,LOG_WARN)("BlitSurface returned %d",Blit);
			} else {
				SDL_UnlockSurface(sdl.surface);
			}
	if(changedLines && (changedLines[0] == sdl.draw.height)) 
	return; 
	if(!menu.hidecycles) frames++;
			SDL_Flip(sdl.surface);
		} else if (changedLines) {
	if(changedLines[0] == sdl.draw.height) 
	return; 
	if(!menu.hidecycles) frames++;
			Bitu y = 0, index = 0, rectCount = 0;
			while (y < sdl.draw.height) {
				if (!(index & 1)) {
					y += changedLines[index];
				} else {
					SDL_Rect *rect = &sdl.updateRects[rectCount++];
					rect->x = sdl.clip.x;
					rect->y = sdl.clip.y + y;
					rect->w = (Bit16u)sdl.draw.width;
					rect->h = changedLines[index];
#if 0
					if (rect->h + rect->y > sdl.surface->h) {
						LOG_MSG("WTF %d +  %d  >%d",rect->h,rect->y,sdl.surface->h);
					}
#endif
					y += changedLines[index];
				}
				index++;
			}
			if (rectCount)
				SDL_UpdateRects( sdl.surface, rectCount, sdl.updateRects );
		}
		break;
#if (HAVE_DDRAW_H) && defined(WIN32)
	case SCREEN_SURFACE_DDRAW:
		SDL_UnlockSurface(sdl.blit.surface);
	if(changedLines && (changedLines[0] == sdl.draw.height)) 
	return; 
	if(!menu.hidecycles) frames++;
		ret=IDirectDrawSurface3_Blt(
			sdl.surface->hwdata->dd_writebuf,&sdl.blit.rect,
			sdl.blit.surface->hwdata->dd_surface,0,
			DDBLT_WAIT, NULL);
		switch (ret) {
		case DD_OK:
			break;
		case DDERR_SURFACELOST:
			IDirectDrawSurface3_Restore(sdl.blit.surface->hwdata->dd_surface);
			IDirectDrawSurface3_Restore(sdl.surface->hwdata->dd_surface);
			break;
		default:
			LOG_MSG("DDRAW:Failed to blit, error %X",ret);
		}
		SDL_Flip(sdl.surface);
		break;
#endif
	case SCREEN_OVERLAY:
		SDL_UnlockYUVOverlay(sdl.overlay);
	if(changedLines && (changedLines[0] == sdl.draw.height)) 
	return; 
	if(!menu.hidecycles) frames++; 
		SDL_DisplayYUVOverlay(sdl.overlay,&sdl.clip);
		break;
#if C_OPENGL
	case SCREEN_OPENGL:
#if defined(NVIDIA_PixelDataRange)
		if (sdl.opengl.pixel_data_range) {
	if(changedLines && (changedLines[0] == sdl.draw.height)) 
	return; 
	glBindTexture(GL_TEXTURE_2D, sdl.opengl.texture);
			glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0,
					sdl.draw.width, sdl.draw.height, GL_BGRA_EXT,
					GL_UNSIGNED_INT_8_8_8_8_REV, sdl.opengl.framebuf);
			glCallList(sdl.opengl.displaylist);
		} else
#endif
		if (changedLines) {
	if(changedLines[0] == sdl.draw.height) 
	return;
			Bitu y = 0, index = 0;
            glBindTexture(GL_TEXTURE_2D, sdl.opengl.texture);
			while (y < sdl.draw.height) {
				if (!(index & 1)) {
					y += changedLines[index];
				} else {
					Bit8u *pixels = (Bit8u *)sdl.opengl.framebuf + y * sdl.opengl.pitch;
					Bitu height = changedLines[index];
					glTexSubImage2D(GL_TEXTURE_2D, 0, 0, y,
						sdl.draw.width, height, GL_BGRA_EXT,
						GL_UNSIGNED_INT_8_8_8_8_REV, pixels );
					y += height;
				}
				index++;
			}
			glCallList(sdl.opengl.displaylist);
		}
	if(!menu.hidecycles) frames++; 
	SDL_GL_SwapBuffers();
		break;
#endif
#if (HAVE_D3D9_H) && defined(WIN32)
	case SCREEN_DIRECT3D:
		if(!menu.hidecycles) frames++; //implemented
		if(GCC_UNLIKELY(!d3d->UnlockTexture(changedLines))) {
			E_Exit("Failed to draw screen!");
		}
		break;
#endif
	default:
		break;
	}
}


void GFX_SetPalette(Bitu start,Bitu count,GFX_PalEntry * entries) {
	/* I should probably not change the GFX_PalEntry :) */
	if (sdl.surface->flags & SDL_HWPALETTE) {
		if (!SDL_SetPalette(sdl.surface,SDL_PHYSPAL,(SDL_Color *)entries,start,count)) {
			E_Exit("SDL:Can't set palette");
		}
	} else {
		if (!SDL_SetPalette(sdl.surface,SDL_LOGPAL,(SDL_Color *)entries,start,count)) {
			E_Exit("SDL:Can't set palette");
		}
	}
}

Bitu GFX_GetRGB(Bit8u red,Bit8u green,Bit8u blue) {
	switch (sdl.desktop.type) {
	case SCREEN_SURFACE:
	case SCREEN_SURFACE_DDRAW:
		return SDL_MapRGB(sdl.surface->format,red,green,blue);
	case SCREEN_OVERLAY:
		{
			Bit8u y =  ( 9797*(red) + 19237*(green) +  3734*(blue) ) >> 15;
			Bit8u u =  (18492*((blue)-(y)) >> 15) + 128;
			Bit8u v =  (23372*((red)-(y)) >> 15) + 128;
#ifdef WORDS_BIGENDIAN
			return (y << 0) | (v << 8) | (y << 16) | (u << 24);
#else
			return (u << 0) | (y << 8) | (v << 16) | (y << 24);
#endif
		}
	case SCREEN_OPENGL:
//		return ((red << 0) | (green << 8) | (blue << 16)) | (255 << 24);
		//USE BGRA
		return ((blue << 0) | (green << 8) | (red << 16)) | (255 << 24);
	case SCREEN_DIRECT3D:
#if (HAVE_D3D9_H) && defined(WIN32)
		if(GCC_UNLIKELY(d3d->bpp16))
		    return SDL_MapRGB(sdl.surface->format,red,green,blue);
		else
#endif
		    return ((blue << 0) | (green << 8) | (red << 16)) | (255 << 24);
	}
	return 0;
}

void GFX_Stop() {
	if (sdl.updating)
		GFX_EndUpdate( 0 );
	sdl.active=false;
}

void GFX_Start() {
	sdl.active=true;
}

static void GUI_ShutDown(Section * /*sec*/) {
	GFX_Stop();
	if (sdl.draw.callback) (sdl.draw.callback)( GFX_CallBackStop );
	if (sdl.mouse.locked) GFX_CaptureMouse();
	if (sdl.desktop.fullscreen) GFX_SwitchFullScreen();
#if (HAVE_D3D9_H) && defined(WIN32)
	if ((sdl.desktop.type==SCREEN_DIRECT3D) && (d3d))
		delete d3d;
#endif
}


static void SetPriority(PRIORITY_LEVELS level) {

#if C_SET_PRIORITY
// Do nothing if priorties are not the same and not root, else the highest
// priority can not be set as users can only lower priority (not restore it)

	if((sdl.priority.focus != sdl.priority.nofocus ) &&
		(getuid()!=0) ) return;

#endif
	switch (level) {
#ifdef WIN32
	case PRIORITY_LEVEL_PAUSE:	// if DOSBox is paused, assume idle priority
	case PRIORITY_LEVEL_LOWEST:
		SetPriorityClass(GetCurrentProcess(),IDLE_PRIORITY_CLASS);
		break;
	case PRIORITY_LEVEL_LOWER:
		SetPriorityClass(GetCurrentProcess(),BELOW_NORMAL_PRIORITY_CLASS);
		break;
	case PRIORITY_LEVEL_NORMAL:
		SetPriorityClass(GetCurrentProcess(),NORMAL_PRIORITY_CLASS);
		break;
	case PRIORITY_LEVEL_HIGHER:
		SetPriorityClass(GetCurrentProcess(),ABOVE_NORMAL_PRIORITY_CLASS);
		break;
	case PRIORITY_LEVEL_HIGHEST:
		SetPriorityClass(GetCurrentProcess(),HIGH_PRIORITY_CLASS);
		break;
#elif C_SET_PRIORITY
/* Linux use group as dosbox has mulitple threads under linux */
	case PRIORITY_LEVEL_PAUSE:	// if DOSBox is paused, assume idle priority
	case PRIORITY_LEVEL_LOWEST:
		setpriority (PRIO_PGRP, 0,PRIO_MAX);
		break;
	case PRIORITY_LEVEL_LOWER:
		setpriority (PRIO_PGRP, 0,PRIO_MAX-(PRIO_TOTAL/3));
		break;
	case PRIORITY_LEVEL_NORMAL:
		setpriority (PRIO_PGRP, 0,PRIO_MAX-(PRIO_TOTAL/2));
		break;
	case PRIORITY_LEVEL_HIGHER:
		setpriority (PRIO_PGRP, 0,PRIO_MAX-((3*PRIO_TOTAL)/5) );
		break;
	case PRIORITY_LEVEL_HIGHEST:
		setpriority (PRIO_PGRP, 0,PRIO_MAX-((3*PRIO_TOTAL)/4) );
		break;
#endif
	default:
		break;
	}
}

extern Bit8u int10_font_14[256 * 14];
static void OutputString(Bitu x,Bitu y,const char * text,Bit32u color,Bit32u color2,SDL_Surface * output_surface) {
	Bit32u * draw=(Bit32u*)(((Bit8u *)output_surface->pixels)+((y)*output_surface->pitch))+x;
	while (*text) {
		Bit8u * font=&int10_font_14[(*text)*14];
		Bitu i,j;
		Bit32u * draw_line=draw;
		for (i=0;i<14;i++) {
			Bit8u map=*font++;
			for (j=0;j<8;j++) {
				if (map & 0x80) *((Bit32u*)(draw_line+j))=color; else *((Bit32u*)(draw_line+j))=color2;
				map<<=1;
			}
			draw_line+=output_surface->pitch/4;
		}
		text++;
		draw+=8;
	}
}

#include "dosbox_splash.h"

/* The endian part is intentionally disabled as somehow it produces correct results without according to rhoenie*/
//#if SDL_BYTEORDER == SDL_BIG_ENDIAN
//    Bit32u rmask = 0xff000000;
//    Bit32u gmask = 0x00ff0000;
//    Bit32u bmask = 0x0000ff00;
//#else
    Bit32u rmask = 0x000000ff;
    Bit32u gmask = 0x0000ff00;
    Bit32u bmask = 0x00ff0000;
//#endif

static SDL_Surface* splash_surf;
static bool			splash_active;
static Bit8u*		splash_tmpbuf;
static Bit32u		splash_startticks;

static void ShowSplashScreen() {
	splash_surf = SDL_CreateRGBSurface(SDL_SWSURFACE, 640, 400, 32, rmask, gmask, bmask, 0);
	if (splash_surf) {
		splash_active=true;
		SDL_FillRect(splash_surf, NULL, SDL_MapRGB(splash_surf->format, 0, 0, 0));
		splash_tmpbuf = new Bit8u[640*400*3];
		GIMP_IMAGE_RUN_LENGTH_DECODE(splash_tmpbuf,gimp_image.rle_pixel_data,640*400,3);
		for (Bitu y=0; y<400; y++) {

			Bit8u* tmpbuf = splash_tmpbuf + y*640*3;
			Bit32u * draw=(Bit32u*)(((Bit8u *)splash_surf->pixels)+((y)*splash_surf->pitch));
			for (Bitu x=0; x<640; x++) {
//#if SDL_BYTEORDER == SDL_BIG_ENDIAN
//				*draw++ = tmpbuf[x*3+2]+tmpbuf[x*3+1]*0x100+tmpbuf[x*3+0]*0x10000+0x00000000;
//#else
				*draw++ = tmpbuf[x*3+0]+tmpbuf[x*3+1]*0x100+tmpbuf[x*3+2]*0x10000+0x00000000;
//#endif
			}
		}
		Bit32u lasttick=GetTicks();
		for(Bitu i = 0; i <=5; i++) {
			if((GetTicks()-lasttick)>20) i++;
			while((GetTicks()-lasttick)<15) SDL_Delay(5);
			lasttick = GetTicks();
			SDL_SetAlpha(splash_surf, SDL_SRCALPHA,(Bit8u)(51*i));
			SDL_BlitSurface(splash_surf, NULL, sdl.surface, NULL);
			SDL_Flip(sdl.surface);
		}

		splash_startticks=GetTicks();
	} else {
		splash_active=false;
		splash_startticks=0;

	}
}

static void EndSplashScreen() {
	if(!splash_active) return;
	//SDL_FillRect(splash_surf, NULL, SDL_MapRGB(sdl.surface->format, 0, 0, 0));
	//SDL_BlitSurface(splash_surf, NULL, sdl.surface, NULL);
	//SDL_Flip(sdl.surface);
	while((GetTicks()-splash_startticks)< 500) SDL_Delay(10);
	
	SDL_FreeSurface(splash_surf);
	delete [] splash_tmpbuf;
	splash_active=false;

}

#if (HAVE_D3D9_H) && defined(WIN32)
#include "SDL_syswm.h"
#endif

void Restart(bool pressed);

static void GUI_StartUp(Section * sec) {
	sec->AddDestroyFunction(&GUI_ShutDown);
	Section_prop * section=static_cast<Section_prop *>(sec);
	sdl.active=false;
	sdl.updating=false;

	GFX_SetIcon();

	sdl.desktop.lazy_fullscreen=false;
	sdl.desktop.lazy_fullscreen_req=false;

	sdl.desktop.fullscreen=section->Get_bool("fullscreen");
	sdl.wait_on_error=section->Get_bool("waitonerror");

	Prop_multival* p=section->Get_multival("priority");
	std::string focus = p->GetSection()->Get_string("active");
	std::string notfocus = p->GetSection()->Get_string("inactive");

	if      (focus == "lowest")  { sdl.priority.focus = PRIORITY_LEVEL_LOWEST;  }
	else if (focus == "lower")   { sdl.priority.focus = PRIORITY_LEVEL_LOWER;   }
	else if (focus == "normal")  { sdl.priority.focus = PRIORITY_LEVEL_NORMAL;  }
	else if (focus == "higher")  { sdl.priority.focus = PRIORITY_LEVEL_HIGHER;  }
	else if (focus == "highest") { sdl.priority.focus = PRIORITY_LEVEL_HIGHEST; }

	if      (notfocus == "lowest")  { sdl.priority.nofocus=PRIORITY_LEVEL_LOWEST;  }
	else if (notfocus == "lower")   { sdl.priority.nofocus=PRIORITY_LEVEL_LOWER;   }
	else if (notfocus == "normal")  { sdl.priority.nofocus=PRIORITY_LEVEL_NORMAL;  }
	else if (notfocus == "higher")  { sdl.priority.nofocus=PRIORITY_LEVEL_HIGHER;  }
	else if (notfocus == "highest") { sdl.priority.nofocus=PRIORITY_LEVEL_HIGHEST; }
	else if (notfocus == "pause")   {
		/* we only check for pause here, because it makes no sense
		 * for DOSBox to be paused while it has focus
		 */
		sdl.priority.nofocus=PRIORITY_LEVEL_PAUSE;
	}

	SetPriority(sdl.priority.focus); //Assume focus on startup
	sdl.mouse.locked=false;
	mouselocked=false; //Global for mapper
	sdl.mouse.requestlock=false;
	sdl.desktop.full.fixed=false;
	const char* fullresolution=section->Get_string("fullresolution");
	sdl.desktop.full.width  = 0;
	sdl.desktop.full.height = 0;
	if(fullresolution && *fullresolution) {
		char res[100];
		safe_strncpy( res, fullresolution, sizeof( res ));
		fullresolution = lowcase (res);//so x and X are allowed
		if(strcmp(fullresolution,"original")) {
			sdl.desktop.full.fixed = true;
			char* height = const_cast<char*>(strchr(fullresolution,'x'));
			if(height && * height) {
				*height = 0;
				sdl.desktop.full.height = (Bit16u)atoi(height+1);
				sdl.desktop.full.width  = (Bit16u)atoi(res);
			}
		}
	}

	sdl.desktop.window.width  = 0;
	sdl.desktop.window.height = 0;
	const char* windowresolution=section->Get_string("windowresolution");
	if(windowresolution && *windowresolution) {
		char res[100];
		safe_strncpy( res,windowresolution, sizeof( res ));
		windowresolution = lowcase (res);//so x and X are allowed
		if(strcmp(windowresolution,"original")) {
			char* height = const_cast<char*>(strchr(windowresolution,'x'));
			if(height && *height) {
				*height = 0;
				sdl.desktop.window.height = (Bit16u)atoi(height+1);
				sdl.desktop.window.width  = (Bit16u)atoi(res);
			}
		}
	}
	sdl.desktop.doublebuf=section->Get_bool("fulldouble");
	int width, height;
	SDL_GetDesktopMode(&width, &height);
	if (!sdl.desktop.full.width) {
		sdl.desktop.full.width=width;
	}
	if (!sdl.desktop.full.height) {
		sdl.desktop.full.height=width;
	}
	sdl.mouse.autoenable=section->Get_bool("autolock");
	if (!sdl.mouse.autoenable) SDL_ShowCursor(SDL_DISABLE);
	sdl.mouse.autolock=false;
	sdl.mouse.sensitivity=section->Get_int("sensitivity");
	std::string output=section->Get_string("output");

	/* Setup Mouse correctly if fullscreen */
	if(sdl.desktop.fullscreen) GFX_CaptureMouse();

	if (output == "surface") {
		sdl.desktop.want_type=SCREEN_SURFACE;
#if (HAVE_DDRAW_H) && defined(WIN32)
	} else if (output == "ddraw") {
		sdl.desktop.want_type=SCREEN_SURFACE_DDRAW;
#endif
	} else if (output == "overlay") {
		sdl.desktop.want_type=SCREEN_OVERLAY;
#if C_OPENGL
	} else if (output == "opengl") {
		sdl.desktop.want_type=SCREEN_OPENGL;
		sdl.opengl.bilinear=true;
	} else if (output == "openglnb") {
		sdl.desktop.want_type=SCREEN_OPENGL;
		sdl.opengl.bilinear=false;
#endif
#if (HAVE_D3D9_H) && defined(WIN32)
	} else if (output == "direct3d") {
		sdl.desktop.want_type=SCREEN_DIRECT3D;
#if LOG_D3D
		LOG_MSG("SDL:Direct3D activated");
#endif
#endif
	} else if (output == "openglhq") {
		char *oldvideo = getenv("SDL_VIDEODRIVER");

		if (oldvideo && strcmp(oldvideo,"openglhq")) {
		    char *driver = (char *)malloc(strlen(oldvideo)+strlen("SDL_OPENGLHQ_VIDEODRIVER=")+1);
		    strcpy(driver,"SDL_OPENGLHQ_VIDEODRIVER=");
		    strcat(driver,oldvideo);
		    putenv(driver);
		    free(driver);
		}
		if (sdl.desktop.doublebuf) putenv("SDL_OPENGLHQ_DOUBLEBUF=1");
		SDL_QuitSubSystem(SDL_INIT_VIDEO);
		putenv("SDL_VIDEODRIVER=openglhq");
		SDL_InitSubSystem(SDL_INIT_VIDEO);
		sdl.desktop.want_type=SCREEN_OPENGLHQ;

	} else {
		LOG_MSG("SDL:Unsupported output device %s, switching back to surface",output.c_str());
		sdl.desktop.want_type=SCREEN_SURFACE;//SHOULDN'T BE POSSIBLE anymore
	}
	sdl.overscan_width=section->Get_int("overscan");
//	sdl.overscan_color=section->Get_int("overscancolor");

	sdl.overlay=0;
#if C_OPENGL
   if(sdl.desktop.want_type==SCREEN_OPENGL){ /* OPENGL is requested */
	sdl.surface=SDL_SetVideoMode(640,400,0,SDL_OPENGL);
	if (sdl.surface == NULL) {
		LOG_MSG("Could not initialize OpenGL, switching back to surface");
		sdl.desktop.want_type=SCREEN_SURFACE;
	} else {
	sdl.opengl.framebuf=0;
	sdl.opengl.texture=0;
	sdl.opengl.displaylist=0;
	glGetIntegerv (GL_MAX_TEXTURE_SIZE, &sdl.opengl.max_texsize);
#if defined(__WIN32__) && defined(NVIDIA_PixelDataRange)
	glPixelDataRangeNV = (PFNGLPIXELDATARANGENVPROC) wglGetProcAddress("glPixelDataRangeNV");
	db_glAllocateMemoryNV = (PFNWGLALLOCATEMEMORYNVPROC) wglGetProcAddress("wglAllocateMemoryNV");
	db_glFreeMemoryNV = (PFNWGLFREEMEMORYNVPROC) wglGetProcAddress("wglFreeMemoryNV");
#endif
	const char * gl_ext = (const char *)glGetString (GL_EXTENSIONS);
	if(gl_ext && *gl_ext){
		sdl.opengl.packed_pixel=(strstr(gl_ext,"EXT_packed_pixels") > 0);
		sdl.opengl.paletted_texture=(strstr(gl_ext,"EXT_paletted_texture") > 0);
#if defined(NVIDIA_PixelDataRange)
		sdl.opengl.pixel_data_range=(strstr(gl_ext,"GL_NV_pixel_data_range") >0 ) &&
			glPixelDataRangeNV && db_glAllocateMemoryNV && db_glFreeMemoryNV;
		sdl.opengl.pixel_data_range = 0;
#endif
    	} else {
		sdl.opengl.packed_pixel=sdl.opengl.paletted_texture=false;
	}
	}
	} /* OPENGL is requested end */

#endif	//OPENGL
	/* Initialize screen for first time */
	sdl.surface=SDL_SetVideoMode(640,400,0,SDL_RESIZABLE);
	if (sdl.surface == NULL) E_Exit("Could not initialize video: %s",SDL_GetError());
	sdl.desktop.bpp=sdl.surface->format->BitsPerPixel;
	if (sdl.desktop.bpp==24) {
		LOG_MSG("SDL:You are running in 24 bpp mode, this will slow down things!");
	}	
#if (HAVE_D3D9_H) && defined(WIN32)
	if(sdl.desktop.want_type==SCREEN_DIRECT3D) {
	    SDL_SysWMinfo wmi;
	    SDL_VERSION(&wmi.version);

	    if(!SDL_GetWMInfo(&wmi)) {
		LOG_MSG("SDL:Error retrieving window information");
		E_Exit("Failed to get window info");
	    }

	    if(d3d) delete d3d;
	    d3d = new CDirect3D(640,400);

	    if(!d3d) E_Exit("Failed to create d3d object");

	    if(d3d->InitializeDX(wmi.window,sdl.desktop.doublebuf) != S_OK)
		    E_Exit("Unable to initialize DirectX");
	}
#endif

	GFX_Stop();
	SDL_WM_SetCaption("DOSBox",VERSION);

	/* Please leave the Splash screen stuff in working order in DOSBox. We spend a lot of time making DOSBox. */
	ShowSplashScreen();

	/* Get some Event handlers */
#ifdef __WIN32__
	MAPPER_AddHandler(ToggleMenu,MK_return,MMOD1|MMOD2,"togglemenu","ToggleMenu");
#endif // WIN32
	MAPPER_AddHandler(KillSwitch,MK_f9,MMOD1,"shutdown","ShutDown");
	MAPPER_AddHandler(CaptureMouse,MK_f10,MMOD1,"capmouse","Cap Mouse");
	MAPPER_AddHandler(SwitchFullScreen,MK_return,MMOD2,"fullscr","Fullscreen");
	MAPPER_AddHandler(Restart,MK_home,MMOD1|MMOD2,"restart","Restart");
#if C_DEBUG
	/* Pause binds with activate-debugger */
#else
	MAPPER_AddHandler(&PauseDOSBox, MK_pause, MMOD2, "pause", "Pause");
#endif
	MAPPER_AddHandler(&UI_Run, MK_f10, MMOD2, "gui", "ShowGUI");
	/* Get Keyboard state of numlock and capslock */
	SDLMod keystate = SDL_GetModState();
	if(keystate&KMOD_NUM) startup_state_numlock = true;
	if(keystate&KMOD_CAPS) startup_state_capslock = true;
}

void Mouse_AutoLock(bool enable) {
	sdl.mouse.autolock=enable;
	if (sdl.mouse.autoenable) sdl.mouse.requestlock=enable;
	else {
		SDL_ShowCursor(enable?SDL_DISABLE:SDL_ENABLE);
		sdl.mouse.requestlock=false;
	}
}

static void RedrawScreen(Bit32u nWidth, Bit32u nHeight) {
   int width;
   int height;
#ifdef __WIN32__
   width=sdl.clip.w; 
   height=sdl.clip.h;
#else
   width=sdl.draw.width; 
   height=sdl.draw.height;
#endif
    void RENDER_CallBack( GFX_CallBackFunctions_t function );
    while (sdl.desktop.fullscreen) {
        int temp_size;
        temp_size=render.scale.size;
        if(!sdl.desktop.fullscreen) { render.scale.size=temp_size; RENDER_CallBack( GFX_CallBackReset); return; }
    }
#ifdef WIN32
    if(menu.resizeusing) {
        RENDER_CallBack( GFX_CallBackReset);
        return;
    }
#endif
	Section_prop * section=static_cast<Section_prop *>(control->GetSection("sdl")); 
	if ((!strcmp(section->Get_string("windowresolution"),"original")) && (render.src.dblw && render.src.dblh)) {
		switch (render.scale.op) {
			case scalerOpNormal:
				if(nWidth>width || nHeight>height) {
                    if (render.scale.size == 1) { ++render.scale.size; break; }
                    if (render.scale.size == 2) { ++render.scale.size; break; }
                    if (render.scale.size == 4) { render.scale.size=6; break; }
                }
                if(nWidth<width || nHeight<height) {
                    if (render.scale.size == 6) { render.scale.size=4; break; }
                    if (render.scale.size == 3) { --render.scale.size; break; }
                    if (render.scale.size == 2) { --render.scale.size; break; }
                }
				break;
			case scalerOpAdvMame:
			case scalerOpHQ:
			case scalerOpAdvInterp:
			case scalerOpTV:
			case scalerOpRGB:
			case scalerOpScan:
				if(nWidth>width || nHeight>height) { if (render.scale.size == 2) ++render.scale.size; }
				if(nWidth<width || nHeight<height) { if (render.scale.size == 3) --render.scale.size; }
				break;
			case scalerOpSaI:
            case scalerOpSuperSaI:
            case scalerOpSuperEagle:
                break;
			default: // other scalers
				break;
		}
	}
    RENDER_CallBack( GFX_CallBackReset);
}

static void HandleVideoResize(void * event) {
	if(sdl.desktop.fullscreen) return;

	SDL_ResizeEvent* ResizeEvent = (SDL_ResizeEvent*)event;
	RedrawScreen(ResizeEvent->w, ResizeEvent->h);
/*    if(sdl.desktop.want_type!=SCREEN_DIRECT3D) {
        HWND hwnd=GetHWND();
        RECT myrect;
        GetClientRect(hwnd,&myrect);
        if(myrect.right==GetSystemMetrics(SM_CXSCREEN)) 
            GFX_SwitchFullScreen();
    } */
#ifdef WIN32
    menu.resizeusing=false;
#endif
}

static void HandleMouseMotion(SDL_MouseMotionEvent * motion) {
	if (sdl.mouse.locked || !sdl.mouse.autoenable)
		Mouse_CursorMoved((float)motion->xrel*sdl.mouse.sensitivity/100.0f,
						  (float)motion->yrel*sdl.mouse.sensitivity/100.0f,
						  (float)(motion->x-sdl.clip.x)/(sdl.clip.w-1)*sdl.mouse.sensitivity/100.0f,
						  (float)(motion->y-sdl.clip.y)/(sdl.clip.h-1)*sdl.mouse.sensitivity/100.0f,
						  sdl.mouse.locked);
}

static void HandleMouseButton(SDL_MouseButtonEvent * button) {
	switch (button->state) {
	case SDL_PRESSED:
		if (sdl.mouse.requestlock && !sdl.mouse.locked) {
			GFX_CaptureMouse();
			// Dont pass klick to mouse handler
			break;
		}
		if (!sdl.mouse.autoenable && sdl.mouse.autolock && button->button == SDL_BUTTON_MIDDLE) {
			GFX_CaptureMouse();
			break;
		}
		switch (button->button) {
		case SDL_BUTTON_LEFT:
			Mouse_ButtonPressed(0);
			break;
		case SDL_BUTTON_RIGHT:
			Mouse_ButtonPressed(1);
			break;
		case SDL_BUTTON_MIDDLE:
			Mouse_ButtonPressed(2);
			break;
		}
		break;
	case SDL_RELEASED:
		switch (button->button) {
		case SDL_BUTTON_LEFT:
			Mouse_ButtonReleased(0);
			break;
		case SDL_BUTTON_RIGHT:
			Mouse_ButtonReleased(1);
			break;
		case SDL_BUTTON_MIDDLE:
			Mouse_ButtonReleased(2);
			break;
		}
		break;
	}
}

void GFX_LosingFocus(void) {
	sdl.laltstate=SDL_KEYUP;
	sdl.raltstate=SDL_KEYUP;
	MAPPER_LosingFocus();
}

bool GFX_IsFullscreen(void) {
	return sdl.desktop.fullscreen;
}

#ifdef __WIN32__
void OpenFileDialog( char * path_arg ) {
	if(control->SecureMode()) {
		LOG_MSG(MSG_Get("PROGRAM_CONFIG_SECURE_DISALLOW"));
		return;
	}
	DOS_MCB mcb(dos.psp()-1);
	static char pcname[9];
	mcb.GetFileName(pcname);
	if(strlen(pcname)) return;

	char message[150];
	OPENFILENAME OpenFileName;
	char szFile[MAX_PATH];
	char CurrentDir[MAX_PATH];
	const char * Temp_CurrentDir = CurrentDir;

	if (Drives['C'-'A']) {
		sprintf(message,"Quick launch automatically mounts drive C in DOSBox.\nDrive C has already been mounted. Do you want to continue?");
		if (MessageBox(GetHWND(),message,"Warning",MB_YESNO)==IDNO) return;
	}

	if(path_arg) goto search;
	szFile[0] = 0;

	GetCurrentDirectory( MAX_PATH, CurrentDir );

	OpenFileName.lStructSize = sizeof( OPENFILENAME );
	OpenFileName.hwndOwner = NULL;
	if(DOSBox_Kor())
		OpenFileName.lpstrFilter = " (*.com, *.exe, *.bat)\0*.com;*.exe;*.bat\0 (*.*)\0*.*\0";
	else
		OpenFileName.lpstrFilter = "Executable files(*.com, *.exe, *.bat)\0*.com;*.exe;*.bat\0All files(*.*)\0*.*\0";
	OpenFileName.lpstrCustomFilter = NULL;
	OpenFileName.nMaxCustFilter = 0;
	OpenFileName.nFilterIndex = 0;
	OpenFileName.lpstrFile = szFile;
	OpenFileName.nMaxFile = sizeof( szFile );
	OpenFileName.lpstrFileTitle = NULL;
	OpenFileName.nMaxFileTitle = 0;
	OpenFileName.lpstrInitialDir = CurrentDir;
	OpenFileName.lpstrTitle = "Select an executable";
	OpenFileName.nFileOffset = 0;
	OpenFileName.nFileExtension = 0;
	OpenFileName.lpstrDefExt = NULL;
	OpenFileName.lCustData = 0;
	OpenFileName.lpfnHook = NULL;
	OpenFileName.lpTemplateName = NULL;
	OpenFileName.Flags = OFN_EXPLORER;

search:
	if(GetOpenFileName( &OpenFileName ) || path_arg) {
		WIN32_FIND_DATA FindFileData;
		HANDLE hFind;
		char drive	[_MAX_DRIVE]; 
		char dir	[_MAX_DIR]; 
		char fname	[_MAX_FNAME]; 
		char ext	[_MAX_EXT]; 
		char * path = 0;
		if(path_arg) {
			szFile[0] = 0;
			sprintf(szFile,path_arg);
		}
		path = szFile;
		_splitpath (path, drive, dir, fname, ext);
		char ext_temp [_MAX_EXT]; ext_temp[0] = 0; sprintf(ext_temp,ext);

		hFind = FindFirstFile(szFile, &FindFileData);
		if (hFind == INVALID_HANDLE_VALUE) {
			if(strcasecmp(ext,"")) goto search;
			szFile[0] = 0;
			ext[0] = 0; sprintf(ext,".com");
			sprintf(szFile,"%s%s%s%s",drive,dir,fname,".com");
			hFind = FindFirstFile(szFile, &FindFileData);
			if (hFind == INVALID_HANDLE_VALUE) {
				szFile[0] = 0;
				ext[0] = 0; sprintf(ext,".exe");
				sprintf(szFile,"%s%s%s%s",drive,dir,fname,".exe");
				hFind = FindFirstFile(szFile, &FindFileData);
				if (hFind == INVALID_HANDLE_VALUE) {
					szFile[0] = 0;
					ext[0] = 0; sprintf(ext,".bat");
					sprintf(szFile,"%s%s%s%s",drive,dir,fname,".bat");
					hFind = FindFirstFile(szFile, &FindFileData);
					if (hFind == INVALID_HANDLE_VALUE) {
						szFile[0] = 0;
						ext[0]=0;
						goto search;
					}
				}
			}
		}

		if((!strcasecmp(ext,".com")) || (!strcasecmp(ext,".exe")) || (!strcasecmp(ext,".bat"))) {
			char pathname[DOS_PATHLENGTH];
			sprintf(pathname,"%s%s",drive,dir);
			MountDrive_2('C',pathname,"LOCAL");
			DOS_SetDrive(toupper('C') - 'A');
		} else {
			LOG_MSG("GUI: Unsupported filename extension.");
			goto godefault;
		}

		#define DOSNAMEBUF 256
		char name1[DOSNAMEBUF+1];
		sprintf(name1,"%s%s",fname,ext);
		Bit16u n=1; Bit8u c='\n';
		DOS_WriteFile(STDOUT,&c,&n);

		DOS_Shell shell;
		DOS_MCB mcb(dos.psp()-1);
		static char name[9];
		mcb.GetFileName(name);

		SetCurrentDirectory( Temp_CurrentDir );
		do {
           char temp[256]=" ";
			shell.Execute(name1,temp);
			if(!strcasecmp(ext,".bat")) shell.RunInternal();
			if (!strlen(name)) break;
		} while (1);

		if(strcasecmp(ext,".bat")) DOS_WriteFile(STDOUT,&c,&n);
		shell.ShowPrompt();
	}

godefault:
	SetCurrentDirectory( Temp_CurrentDir );
	return;
}

void Go_Boot(const char boot_drive[_MAX_DRIVE]) {
		if(control->SecureMode()) {
			LOG_MSG(MSG_Get("PROGRAM_CONFIG_SECURE_DISALLOW"));
			return;
		}

	OPENFILENAME OpenFileName;
	char szFile[MAX_PATH];
	char CurrentDir[MAX_PATH];
	const char * Temp_CurrentDir = CurrentDir;
	szFile[0] = 0;
	GetCurrentDirectory( MAX_PATH, CurrentDir );

	OpenFileName.lStructSize = sizeof( OPENFILENAME );
	OpenFileName.hwndOwner = NULL;

	if(DOSBox_Kor())
		OpenFileName.lpstrFilter = "̹ (*.img, *.ima, *.pcjr, *.jrc)\0*.pcjr;*.img;*.ima;*.jrc\0 (*.*)\0*.*\0";
	else
		OpenFileName.lpstrFilter = "Image files(*.img, *.ima, *.pcjr, *.jrc)\0*.pcjr;*.img;*.ima;*.jrc\0All files(*.*)\0*.*\0";

	OpenFileName.lpstrCustomFilter = NULL;
	OpenFileName.nMaxCustFilter = 0;
	OpenFileName.nFilterIndex = 0;
	OpenFileName.lpstrFile = szFile;
	OpenFileName.nMaxFile = sizeof( szFile );
	OpenFileName.lpstrFileTitle = NULL;
	OpenFileName.nMaxFileTitle = 0;
	OpenFileName.lpstrInitialDir = CurrentDir;
	OpenFileName.lpstrTitle = "Select an image file";
	OpenFileName.nFileOffset = 0;
	OpenFileName.nFileExtension = 0;
	OpenFileName.lpstrDefExt = NULL;
	OpenFileName.lCustData = 0;
	OpenFileName.lpfnHook = NULL;
	OpenFileName.lpTemplateName = NULL;
	OpenFileName.Flags = OFN_EXPLORER;
search:
	if(GetOpenFileName( &OpenFileName )) {
		WIN32_FIND_DATA FindFileData;
		HANDLE hFind;
		char drive	[_MAX_DRIVE]; 
		char dir	[_MAX_DIR]; 
		char fname	[_MAX_FNAME]; 
		char ext	[_MAX_EXT]; 
		char * path = 0;
		path = szFile;
		_splitpath (path, drive, dir, fname, ext);
		char ext_temp [_MAX_EXT]; ext_temp[0] = 0; sprintf(ext_temp,ext);

		hFind = FindFirstFile(szFile, &FindFileData);
		if (hFind == INVALID_HANDLE_VALUE) goto search;

		if((!strcasecmp(ext,".img")) || (!strcasecmp(ext,".pcjr")) || (!strcasecmp(ext,".jrc")) || (!strcasecmp(ext,".ima"))) {
			DOS_Shell shell;
			strcat(szFile," -l ");
			strcat(szFile,boot_drive);
			Bit16u n=1; Bit8u c='\n';
			DOS_WriteFile(STDOUT,&c,&n);
           char temp[256]="z:\\boot.com";
			shell.Execute(temp,szFile);
			shell.RunInternal();
			DOS_WriteFile(STDOUT,&c,&n);
			shell.ShowPrompt(); // if failed
		} else {
			LOG_MSG("GUI: Unsupported filename extension.");
			goto godefault;
		}
	}

godefault:
	SetCurrentDirectory( Temp_CurrentDir );
	return;
}

void Drag_Drop( char * path_arg ) {
		if(control->SecureMode()) {
			LOG_MSG(MSG_Get("PROGRAM_CONFIG_SECURE_DISALLOW"));
			return;
		}
	DOS_MCB mcb(dos.psp()-1);
	static char name[9];
	mcb.GetFileName(name);
	if((!path_arg) || (strlen(name)))  return;
	WIN32_FIND_DATA FindFileData;
	HANDLE hFind;
	char drive	[_MAX_DRIVE]; 
	char dir	[_MAX_DIR]; 
	char fname	[_MAX_FNAME]; 
	char ext	[_MAX_EXT]; 
	char szFile[MAX_PATH];

	szFile[0] = 0;
	sprintf(szFile,path_arg);
	char * path = szFile;
	_splitpath (path, drive, dir, fname, ext);
	char ext_temp [_MAX_EXT];
	ext_temp[0] = 0;
	sprintf(ext_temp,ext);

	hFind = FindFirstFile(szFile, &FindFileData);
	if (hFind == INVALID_HANDLE_VALUE) return;

	if((!strcasecmp(ext,".com")) || (!strcasecmp(ext,".exe")) || (!strcasecmp(ext,".bat")))
		OpenFileDialog(path_arg);
	else
		LOG_MSG("GUI: Unsupported filename extension.");
}

void OpenFileDialog_Img( char drive ) {
		if(control->SecureMode()) {
			LOG_MSG(MSG_Get("PROGRAM_CONFIG_SECURE_DISALLOW"));
			return;
		}
	if (Drives[drive-'A']) {
		LOG_MSG("GUI: Unmount drive %c first, and then try again.",drive);
		return;
	}
	OPENFILENAME OpenFileName;
	char szFile[MAX_PATH];
	char CurrentDir[MAX_PATH];
	const char * Temp_CurrentDir = CurrentDir;

	szFile[0] = 0;
	GetCurrentDirectory( MAX_PATH, CurrentDir );
	OpenFileName.lStructSize = sizeof( OPENFILENAME );
	OpenFileName.hwndOwner = NULL;

	if(DOSBox_Kor())
		OpenFileName.lpstrFilter = "̹/ZIP (*.img, *.iso, *.cue, *.bin, *.mdf, *.zip, *.7z)\0*.img;*.iso;*.mdf;*.zip;*.cue;*.bin;*.7z\0 (*.*)\0*.*\0";
	else
		OpenFileName.lpstrFilter = "Image/Zip files(*.img, *.iso, *.cue, *.bin, *.mdf, *.zip, *.7z)\0*.img;*.iso;*.mdf;*.zip;*.cue;*.bin;*.7z\0All files(*.*)\0*.*\0";

	OpenFileName.lpstrCustomFilter = NULL;
	OpenFileName.nMaxCustFilter = 0;
	OpenFileName.nFilterIndex = 0;
	OpenFileName.lpstrFile = szFile;
	OpenFileName.nMaxFile = sizeof( szFile );
	OpenFileName.lpstrFileTitle = NULL;
	OpenFileName.nMaxFileTitle = 0;
	OpenFileName.lpstrInitialDir = CurrentDir;
	OpenFileName.lpstrTitle = "Select an image file";
	OpenFileName.nFileOffset = 0;
	OpenFileName.nFileExtension = 0;
	OpenFileName.lpstrDefExt = NULL;
	OpenFileName.lCustData = 0;
	OpenFileName.lpfnHook = NULL;
	OpenFileName.lpTemplateName = NULL;
	OpenFileName.Flags = OFN_EXPLORER;

search:
	if(GetOpenFileName( &OpenFileName )) {
		WIN32_FIND_DATA FindFileData;
		HANDLE hFind;
		hFind = FindFirstFile(szFile, &FindFileData);
		if (hFind == INVALID_HANDLE_VALUE) goto search;
		char drive2	[_MAX_DRIVE]; 
		char dir	[_MAX_DIR]; 
		char fname	[_MAX_FNAME]; 
		char ext	[_MAX_EXT]; 
		char * path = szFile;

		_splitpath (path, drive2, dir, fname, ext);

		if((!strcasecmp(ext,".img")) || (!strcasecmp(ext,".iso")) || (!strcasecmp(ext,".cue")) || (!strcasecmp(ext,".bin")) || (!strcasecmp(ext,".mdf"))) {
			if(!strcasecmp(ext,".img")) {
				LOG_MSG(".img is automatically considered as a CD-ROM image.\nTo mount it as floppy or hdd, please use IMGMOUNT command instead.");
			}
			Mount_Img(drive,path);
		} else if(!strcasecmp(ext,".zip") || (!strcasecmp(ext,".7z"))) {
			SetCurrentDirectory( Temp_CurrentDir );
			Mount_Zip(drive,path);
		} else
			LOG_MSG("GUI: Unsupported filename extension.");
	}
	SetCurrentDirectory( Temp_CurrentDir );
}

void D3D_PS(void) {
	OPENFILENAME OpenFileName;
	char szFile[MAX_PATH];
	char CurrentDir[MAX_PATH];
	const char * Temp_CurrentDir = CurrentDir;
	szFile[0] = 0;

	GetCurrentDirectory( MAX_PATH, CurrentDir );

	OpenFileName.lStructSize = sizeof( OPENFILENAME );
	OpenFileName.hwndOwner = NULL;
	if(DOSBox_Kor())
		OpenFileName.lpstrFilter = "ȿ (*.fx)\0*.fx\0 (*.*)\0*.*\0";
	else
		OpenFileName.lpstrFilter = "Effect files(*.fx)\0*.fx\0All files(*.*)\0*.*\0";
	OpenFileName.lpstrCustomFilter = NULL;
	OpenFileName.nMaxCustFilter = 0;
	OpenFileName.nFilterIndex = 0;
	OpenFileName.lpstrFile = szFile;
	OpenFileName.nMaxFile = sizeof( szFile );
	OpenFileName.lpstrFileTitle = NULL;
	OpenFileName.nMaxFileTitle = 0;
	OpenFileName.lpstrInitialDir = ".\\Shaders";;
	OpenFileName.lpstrTitle = "Select an effect file";
	OpenFileName.nFileOffset = 0;
	OpenFileName.nFileExtension = 0;
	OpenFileName.lpstrDefExt = NULL;
	OpenFileName.lCustData = 0;
	OpenFileName.lpfnHook = NULL;
	OpenFileName.lpTemplateName = NULL;
	OpenFileName.Flags = OFN_EXPLORER;

search:
	if(GetOpenFileName( &OpenFileName )) {
		WIN32_FIND_DATA FindFileData;
		HANDLE hFind;
		char drive	[_MAX_DRIVE]; 
		char dir	[_MAX_DIR]; 
		char fname	[_MAX_FNAME]; 
		char ext	[_MAX_EXT]; 
		char * path = 0;
		path = szFile;
		_splitpath (path, drive, dir, fname, ext);

		if(!strcasecmp(ext,".fx")) {
			if(!strcasecmp(fname,"none")) { SetVal("sdl","pixelshader","none"); goto godefault; }
			strcat(fname,".fx");
			SetVal("sdl","pixelshader",fname);
		} else {
			LOG_MSG("GUI: Unsupported filename extension.");
			goto godefault;
		}
	}

godefault:
	SetCurrentDirectory( Temp_CurrentDir );
	return;
}


int Reflect_Menu(void) {
	if(!menu.gui) return 0;
	HMENU m_handle=GetMenu(GetHWND()); if(!m_handle) return 0;
	extern bool Mouse_Drv;
	DOS_MCB mcb(dos.psp()-1);
	static char name[9];
	mcb.GetFileName(name);
for (int slot = 0; slot < 10; slot++) {
		std::string path;
		struct stat status;
		bool Get_Custom_SaveDir(std::string& savedir);
		if(Get_Custom_SaveDir(path)) {
			path+="\\";
		} else {
		// check if save directory exists in current directory
		stat( "save", &status );
		if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			path="./save/";
		} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(path);
			path += "save/";
		}
		}
		std::string temp = path;
		std::string realslot_str;
		std::stringstream out;
		int realslot = slot+1;
		out << realslot;
		realslot_str = out.str();
		temp += realslot_str;
		temp += "/";

		std::string program_temp;
		std::ifstream check_title;
		int length = 8;

		program_temp = temp;
		program_temp += "Program Name";
		check_title.open(program_temp.c_str(), std::ifstream::in);
		if(!check_title.fail()) {
			check_title.seekg (0, std::ios::end);
			length = check_title.tellg();
			check_title.seekg (0, std::ios::beg);
			char * const buffer = (char*)_alloca( (length+1) * sizeof(char));
			check_title.read (buffer, length);
			check_title.close();
			char slotstring[30] = "";
			strcpy(slotstring,realslot_str.c_str());
			strcat(slotstring," ("); buffer[length]='\0';
			strcat(slotstring,buffer);
			strcat(slotstring,")");
			ModifyMenu (m_handle, ID_SAVESTATE_1+slot, MF_BYCOMMAND | MF_STRING, ID_SAVESTATE_1+slot, slotstring);
			ModifyMenu (m_handle, ID_LOADSTATE_1+slot, MF_BYCOMMAND | MF_STRING, ID_LOADSTATE_1+slot, slotstring);
			ModifyMenu (m_handle, ID_REMOVE_STATE_1+slot, MF_BYCOMMAND | MF_STRING, ID_REMOVE_STATE_1+slot, slotstring);
		} else {
			char empty_string[20]="";
			strcat(empty_string,realslot_str.c_str());
			strcat(empty_string,"          ");
			ModifyMenu (m_handle, ID_SAVESTATE_1+slot, MF_BYCOMMAND | MF_STRING, ID_SAVESTATE_1+slot, empty_string);
			ModifyMenu (m_handle, ID_LOADSTATE_1+slot, MF_BYCOMMAND | MF_STRING, ID_LOADSTATE_1+slot, empty_string);
			ModifyMenu (m_handle, ID_REMOVE_STATE_1+slot, MF_BYCOMMAND | MF_STRING, ID_REMOVE_STATE_1+slot, empty_string);
		}
	}
		CheckMenuItem(m_handle,ID_WAITONERR,(sdl.wait_on_error)?MF_CHECKED:MF_STRING);
		EnableMenuItem(m_handle,ID_OPENFILE,(strlen(name) || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_GLIDE_TRUE,(strlen(name) || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_GLIDE_FALSE,(strlen(name) || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_GLIDE_EMU,(strlen(name) || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_GLIDE_SECTION,(strlen(name) || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_NONE,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_BG,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_CZ,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_FR,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_GK,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_GR,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_HR,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_HU,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_IT,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_NL,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_NO,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_PL,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_RU,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_SK,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_SP,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_SU,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_SV,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_BE,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_BR,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_CF,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_DK,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_LA,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_PO,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_SF,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_SG,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_UK,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_US,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_YU,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_FO,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_MK,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_MT,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_PH,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_RO,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_SQ,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_TM,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_TR,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_UX,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_YC,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_DV,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_RH,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_KEY_LH,(strlen(name))?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_BOOT_A,(strlen(name) || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_BOOT_C,(strlen(name) || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_BOOT_D,(strlen(name) || menu.boot)?MF_GRAYED:MF_ENABLED);
		CheckMenuItem(m_handle,ID_MOUSE,Mouse_Drv?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_AUTOCYCLE,(CPU_CycleAutoAdjust)?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_AUTODETER,(CPU_AutoDetermineMode)?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_NORMAL,(!strcasecmp(core_mode, "Normal"))?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_DYNAMIC,(!strcasecmp(core_mode, "Dynamic"))?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_FULL,(!strcasecmp(core_mode, "Full"))?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_SIMPLE,(!strcasecmp(core_mode, "Simple"))?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_AUTO,(!strcasecmp(core_mode, "Auto"))?MF_CHECKED:MF_STRING);

		Section_prop * sec=0;
		sec=static_cast<Section_prop *>(control->GetSection("cpu"));
		const std::string cputype=sec->Get_string("cputype");
		CheckMenuItem(m_handle,ID_CPUTYPE_AUTO,cputype=="auto"?MF_CHECKED:MF_STRING);
   		CheckMenuItem(m_handle,ID_CPUTYPE_386,cputype=="386"?MF_CHECKED:MF_STRING);
   		CheckMenuItem(m_handle,ID_CPUTYPE_386_PREFETCH,cputype=="386_prefetch"?MF_CHECKED:MF_STRING);
   		CheckMenuItem(m_handle,ID_CPUTYPE_486,cputype=="486"?MF_CHECKED:MF_STRING);
   		CheckMenuItem(m_handle,ID_CPUTYPE_PENTIUM,cputype=="pentium"?MF_CHECKED:MF_STRING);
   		CheckMenuItem(m_handle,ID_CPUTYPE_PENTIUM_MMX,cputype=="pentium_mmx"?MF_CHECKED:MF_STRING);

        sec=static_cast<Section_prop *>(control->GetSection("joystick"));
		const std::string joysticktype=sec->Get_string("joysticktype");
		CheckMenuItem(m_handle,ID_JOYSTICKTYPE_AUTO,joysticktype=="auto"?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_JOYSTICKTYPE_2AXIS,joysticktype=="2axis"?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICKTYPE_4AXIS,joysticktype=="4axis"?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICKTYPE_4AXIS_2,joysticktype=="4axis_2"?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICKTYPE_FCS,joysticktype=="fcs"?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICKTYPE_CH,joysticktype=="ch"?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICKTYPE_NONE,joysticktype=="none"?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICK_TIMED,sec->Get_bool("timed")?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICK_AUTOFIRE,sec->Get_bool("autofire")?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICK_SWAP34,sec->Get_bool("swap34")?MF_CHECKED:MF_STRING);
  		CheckMenuItem(m_handle,ID_JOYSTICK_BUTTONWRAP,sec->Get_bool("buttonwrap")?MF_CHECKED:MF_STRING);

		CheckMenuItem(m_handle,ID_ASPECT,(render.aspect)?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_SURFACE,(sdl.desktop.want_type==SCREEN_SURFACE)?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_DDRAW,(sdl.desktop.want_type==SCREEN_SURFACE_DDRAW)?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_DIRECT3D,(sdl.desktop.want_type==SCREEN_DIRECT3D)?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_OVERLAY,(sdl.desktop.want_type==SCREEN_OVERLAY)?MF_CHECKED:MF_STRING);
		if(sdl.desktop.want_type==SCREEN_OPENGL) {
			if(sdl.opengl.bilinear) {
				CheckMenuItem(m_handle,ID_OPENGL,MF_CHECKED);
				CheckMenuItem(m_handle,ID_OPENGLNB,MF_STRING);
			} else {
				CheckMenuItem(m_handle,ID_OPENGL,MF_STRING);
				CheckMenuItem(m_handle,ID_OPENGLNB,MF_CHECKED);
			}
		} else {
				CheckMenuItem(m_handle,ID_OPENGLNB,MF_STRING);
				CheckMenuItem(m_handle,ID_OPENGL,MF_STRING);
		}
		CheckMenuItem(m_handle,ID_OPENGLHQ,(sdl.desktop.want_type==SCREEN_OPENGLHQ)?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_FULLDOUBLE,sdl.desktop.doublebuf?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_AUTOLOCK,sdl.mouse.autoenable?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_HIDECYCL,!menu.hidecycles?MF_CHECKED:MF_STRING);

        sec=static_cast<Section_prop *>(control->GetSection("serial"));
        if(sec) {
            bool serial=false;
			const std::string serial1=sec->Get_string("serial1");
			const std::string serial2=sec->Get_string("serial2");
			const std::string serial3=sec->Get_string("serial3");
			const std::string serial4=sec->Get_string("serial4");
            if(serial1 != "disabled" || serial2 != "disabled" || serial3 != "disabled"
            || serial4 != "disabled") serial=true;
    		CheckMenuItem(m_handle,ID_SERIAL_SECTION,serial?MF_CHECKED:MF_STRING);
        }

        sec=static_cast<Section_prop *>(control->GetSection("parallel"));
        if(sec) {
           //CheckMenuItem(m_handle,ID_DONGLE,sec->Get_bool("dongle")?MF_CHECKED:MF_STRING);
            bool parallel=false;
			const std::string parallel1=sec->Get_string("parallel1");
			const std::string parallel2=sec->Get_string("parallel2");
			const std::string parallel3=sec->Get_string("parallel3");
            if(parallel1 != "disabled" || parallel2 != "disabled" || parallel3 != "disabled")
            parallel=true;
    		CheckMenuItem(m_handle,ID_PARALLEL_SECTION,(parallel || sec->Get_bool("dongle"))?MF_CHECKED:MF_STRING);
        }

        std::ifstream check_title;
	    std::string path;
        std::string real_path;
        struct stat status;
		bool Get_Custom_SaveDir(std::string& savedir);
		if(Get_Custom_SaveDir(path)) {
			path+="\\";
		} else {
        // check if save directory exists in current directory
        stat( "save", &status );
        if ( status.st_mode & S_IFDIR ) {
             // path exists in current directory
             path="./save/";
         } else {
            // makes save to appdata
            Cross::GetPlatformConfigDir(path);
            path += "save/";
        }
		}
	        real_path = path + "1/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_1,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_1,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_1,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_1,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_1,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "2/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_2,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_2,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_2,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_2,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_2,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "3/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_3,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_3,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_3,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_3,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_3,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "4/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_4,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_4,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_4,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_4,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_4,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "5/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_5,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_5,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_5,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_5,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_5,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "6/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_6,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_6,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_6,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_6,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_6,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "7/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_7,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_7,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_7,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_7,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_7,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "8/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_8,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_8,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_8,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_8,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_8,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "9/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_9,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_9,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_9,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_9,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_9,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();
	        real_path = path + "10/Program Name";
	        check_title.open(real_path.c_str(), std::ifstream::in);
	        EnableMenuItem(m_handle,ID_LOADSTATE_10,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_LOADSTATE_10,check_title.fail()?MF_STRING:MF_CHECKED);
	        CheckMenuItem(m_handle,ID_SAVESTATE_10,check_title.fail()?MF_STRING:MF_CHECKED);
	        EnableMenuItem(m_handle,ID_REMOVE_STATE_10,check_title.fail()?MF_GRAYED:MF_ENABLED);
	        CheckMenuItem(m_handle,ID_REMOVE_STATE_10,check_title.fail()?MF_STRING:MF_CHECKED);
	        check_title.close();

        sec=static_cast<Section_prop *>(control->GetSection("printer"));
        if(sec) CheckMenuItem(m_handle,ID_PRINTER_SECTION,sec->Get_bool("printer")?MF_CHECKED:MF_STRING);

		sec=static_cast<Section_prop *>(control->GetSection("sdl"));
		if(sec) {
			const char* windowresolution=sec->Get_string("windowresolution");
			const int sdl_overscan = sec->Get_int("overscan");
			CheckMenuItem(m_handle,ID_OVERSCAN_0,sdl_overscan==0?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_1,sdl_overscan==1?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_2,sdl_overscan==2?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_3,sdl_overscan==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_4,sdl_overscan==4?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_5,sdl_overscan==5?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_6,sdl_overscan==6?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_7,sdl_overscan==7?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_8,sdl_overscan==8?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_9,sdl_overscan==9?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_OVERSCAN_10,sdl_overscan==10?MF_CHECKED:MF_STRING);

			if(windowresolution && *windowresolution) {
				char res[100];
				safe_strncpy( res,windowresolution, sizeof( res ));
				windowresolution = lowcase (res);//so x and X are allowed
				CheckMenuItem(m_handle,ID_USESCANCODES,(sec->Get_bool("usescancodes"))?MF_CHECKED:MF_STRING);

				CheckMenuItem(m_handle,ID_NONE,SCALER_2(scalerOpNormal,1)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_NORMAL2X,SCALER_2(scalerOpNormal,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_NORMAL3X,SCALER_2(scalerOpNormal,3)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_HARDWARE2X,SCALER_2(scalerOpNormal,4)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_HARDWARE3X,SCALER_2(scalerOpNormal,6)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_ADVMAME2X,SCALER_2(scalerOpAdvMame,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_ADVMAME3X,SCALER_2(scalerOpAdvMame,3)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_ADVINTERP2X,SCALER_2(scalerOpAdvInterp,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_ADVINTERP3X,SCALER_2(scalerOpAdvInterp,3)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_HQ2X,SCALER_2(scalerOpHQ,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_HQ3X,SCALER_2(scalerOpHQ,3)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_TV2X,SCALER_2(scalerOpTV,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_TV3X,SCALER_2(scalerOpTV,3)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_SCAN2X,SCALER_2(scalerOpScan,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_SCAN3X,SCALER_2(scalerOpScan,3)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_RGB2X,SCALER_2(scalerOpRGB,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_RGB3X,SCALER_2(scalerOpRGB,3)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_2XSAI,SCALER_2(scalerOpSaI,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_SUPER2XSAI,SCALER_2(scalerOpSuperSaI,2)?MF_CHECKED:MF_STRING);
				CheckMenuItem(m_handle,ID_SUPEREAGLE,SCALER_2(scalerOpSuperEagle,2)?MF_CHECKED:MF_STRING);
				//EnableMenuItem(m_handle,ID_FORCESCALER,SCALER_2(scalerOpNormal,4) || SCALER_2(scalerOpNormal,6)?MF_GRAYED:MF_ENABLED);
				CheckMenuItem(m_handle,ID_FORCESCALER,render.scale.forced?MF_CHECKED:MF_STRING);
			}
			const std::string winres=sec->Get_string("windowresolution");
			const std::string fullres=sec->Get_string("fullresolution");

			if(!(winres=="original" || winres=="desktop" || winres=="0x0" || winres=="160x200" || winres=="320x200" || winres=="640x200" ||
			winres=="640x480" || winres=="720x350" || winres=="720x480" || winres=="720x576" ||
			winres=="800x480" || winres=="800x600" || winres=="848x480" || winres=="852x480" ||
			winres=="1024x512" || winres=="1024x768" || winres=="1152x864" || winres=="1280x768" || winres=="1280x800" ||
			winres=="1280x960" || winres=="1280x1024"))
				CheckMenuItem(m_handle,ID_WINRES_USER,MF_CHECKED);
           else
				CheckMenuItem(m_handle,ID_WINRES_USER,MF_STRING);

			if(!(fullres=="original" || fullres=="desktop" || fullres=="0x0" || fullres=="160x200" || fullres=="320x200" || fullres=="640x200" ||
			fullres=="640x480" || fullres=="720x350" || fullres=="720x480" || fullres=="720x576" ||
			fullres=="800x480" || fullres=="800x600" || fullres=="848x480" || fullres=="852x480" ||
			fullres=="1024x512" || fullres=="1024x768" || fullres=="1152x864" || fullres=="1280x768" || fullres=="1280x800" ||
			fullres=="1280x960" || fullres=="1280x1024" || fullres=="1360x765" || fullres=="1366x768" ||
			fullres=="1400x1050" || fullres=="1680x1024" || fullres=="1680x1050" || fullres=="1600x1200" ||
			fullres=="1792x1344" || fullres=="1800x1440" || fullres=="1856x1393" || fullres=="1920x1200" ||
			fullres=="1920x1440" || fullres=="2048x1536" || fullres=="2560x1600" || fullres=="2560x2048" ||
			fullres=="3200x2048" || fullres=="3200x2400" || fullres=="3840x2400" || fullres=="5120x4096" ||
			fullres=="6400x4096" || fullres=="6400x4800" || fullres=="7680x4320" || fullres=="7680x4800"))
				CheckMenuItem(m_handle,ID_WINFULL_USER,MF_CHECKED);
            else
				CheckMenuItem(m_handle,ID_WINFULL_USER,MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_DESKTOP,(fullres=="desktop") || (fullres=="0x0")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_ORIGINAL,(fullres=="original")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_DESKTOP,(winres=="desktop" || winres=="0x0")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_ORIGINAL,(winres=="original")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_160x200,(winres=="160x200")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_320x200,(winres=="320x200")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_320x240,(winres=="320x240")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_640x200,(winres=="640x200")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_640x480,(winres=="640x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_720x350,(winres=="720x350")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_720x480,(winres=="720x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_720x576,(winres=="720x576")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_800x480,(winres=="800x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_800x600,(winres=="800x600")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_848x480,(winres=="848x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_852x480,(winres=="852x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_1024x512,(winres=="1024x512")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_1024x768,(winres=="1024x768")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_1152x864,(winres=="1152x864")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_1280x768,(winres=="1280x768")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_1280x800,(winres=="1280x800")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_1280x960,(winres=="1280x960")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINRES_1280x1024,(winres=="1280x1024")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_160x200,(fullres=="160x200")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_320x200,(fullres=="320x200")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_320x240,(fullres=="320x240")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_640x200,(fullres=="640x200")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_640x480,(fullres=="640x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_720x350,(fullres=="720x350")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_720x480,(fullres=="720x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_720x576,(fullres=="720x576")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_800x480,(fullres=="800x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_800x600,(fullres=="800x600")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_848x480,(fullres=="848x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_852x480,(fullres=="852x480")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1024x512,(fullres=="1024x512")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1024x768,(fullres=="1024x768")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1152x864,(fullres=="1152x864")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1280x768,(fullres=="1280x768")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1280x800,(fullres=="1280x800")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1280x960,(fullres=="1280x960")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1280x1024,(fullres=="1280x1024")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1360x765,(fullres=="1360x765")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1366x768,(fullres=="1366x768")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1400x1050,(fullres=="1400x1050")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1680x1024,(fullres=="1680x1024")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1680x1050,(fullres=="1680x1050")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1600x1200,(fullres=="1600x1200")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1792x1344,(fullres=="1792x1344")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1800x1440,(fullres=="1800x1440")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1856x1393,(fullres=="1856x1393")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1920x1200,(fullres=="1920x1200")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_1920x1440,(fullres=="1920x1440")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_2048x1536,(fullres=="2048x1536")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_2560x1600,(fullres=="2560x1600")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_2560x2048,(fullres=="2560x2048")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_3200x2048,(fullres=="3200x2048")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_3200x2400,(fullres=="3200x2400")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_3840x2400,(fullres=="3840x2400")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_5120x4096,(fullres=="5120x4096")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_6400x4096,(fullres=="6400x4096")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_6400x4800,(fullres=="6400x4800")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_7680x4320,(fullres=="7680x4320")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_WINFULL_7680x4800,(fullres=="7680x4800")?MF_CHECKED:MF_STRING);
		}

		sec=static_cast<Section_prop *>(control->GetSection("midi"));
		if(sec) {
			const std::string mpu401 = sec->Get_string("mpu401");
			const std::string device = sec->Get_string("mididevice");
			const std::string mt32reverbmode = sec->Get_string("mt32.reverb.mode");
			const std::string mt32dac = sec->Get_string("mt32.dac");
			const std::string mt32reversestereo = sec->Get_string("mt32.reverse.stereo");
			const int mt32reverbtime = sec->Get_int("mt32.reverb.time");
			const int mt32reverblevel = sec->Get_int("mt32.reverb.level");
			CheckMenuItem(m_handle,ID_MIDI_NONE,(mpu401=="none")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_UART,(mpu401=="uart")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_INTELLI,(mpu401=="intelligent")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_DEV_NONE,(device=="none")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_DEFAULT,(device=="default")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_ALSA,(device=="alsa")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_OSS,(device=="oss")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_WIN32,(device=="win32")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_COREAUDIO,(device=="coreaudio")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_COREMIDI,(device=="coremidi")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32,(device=="mt32")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_SYNTH,(device=="synth")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_TIMIDITY,(device=="timidity")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBMODE_AUTO,(mt32reverbmode=="auto")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBMODE_0,(mt32reverbmode=="0")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBMODE_1,(mt32reverbmode=="1")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBMODE_2,(mt32reverbmode=="2")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBMODE_3,(mt32reverbmode=="3")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_DAC_AUTO,(mt32dac=="auto")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_DAC_0,(mt32dac=="0")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_DAC_1,(mt32dac=="1")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_DAC_2,(mt32dac=="2")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_DAC_3,(mt32dac=="3")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBTIME_0,mt32reverbtime==0?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBTIME_1,mt32reverbtime==1?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBTIME_2,mt32reverbtime==2?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBTIME_3,mt32reverbtime==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBTIME_4,mt32reverbtime==4?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBTIME_5,mt32reverbtime==5?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBTIME_6,mt32reverbtime==6?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBTIME_7,mt32reverbtime==7?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBLEV_0,mt32reverblevel==0?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBLEV_1,mt32reverblevel==1?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBLEV_2,mt32reverblevel==2?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBLEV_3,mt32reverblevel==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBLEV_4,mt32reverblevel==4?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBLEV_5,mt32reverblevel==5?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBLEV_6,mt32reverblevel==6?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERBLEV_7,mt32reverblevel==7?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERSESTEREO_TRUE,(mt32reversestereo=="on")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MIDI_MT32_REVERSESTEREO_FALSE,(mt32reversestereo=="off")?MF_CHECKED:MF_STRING);
		}

			CheckMenuItem(m_handle,ID_MUTE,menu.nosound?MF_CHECKED:MF_STRING);

		sec=static_cast<Section_prop *>(control->GetSection("mixer"));
		if(sec) {
			CheckMenuItem(m_handle,ID_SWAPSTEREO,sec->Get_bool("swapstereo")?MF_CHECKED:MF_STRING);
		}

		sec=static_cast<Section_prop *>(control->GetSection("sblaster"));
		if(sec) {
			const std::string sbtype = sec->Get_string("sbtype");
			const int sbbase = sec->Get_hex("sbbase");
			const int hwbase = sec->Get_hex("hardwarebase");
			const int irq = sec->Get_int("irq");
			const int dma = sec->Get_int("dma");
			const int hdma = sec->Get_int("hdma");
			const std::string oplmode = sec->Get_string("oplmode");
			const std::string oplemu = sec->Get_string("oplemu");
			const int oplrate = sec->Get_int("oplrate");
			CheckMenuItem(m_handle,ID_SB_NONE,(sbtype=="none")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_SB1,(sbtype=="sb1")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_SB2,(sbtype=="sb2")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_SBPRO1,(sbtype=="sbpro1")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_SBPRO2,(sbtype=="sbpro2")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_SB16,(sbtype=="sb16")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_SB16VIBRA,(sbtype=="sb16vibra")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_GB,(sbtype=="gb")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_220,sbbase==544?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_240,sbbase==576?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_260,sbbase==608?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_280,sbbase==640?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_300,sbbase==768?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_2a0,sbbase==672?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_2c0,sbbase==704?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_2e0,sbbase==736?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HW210,hwbase==528?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HW220,hwbase==544?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HW230,hwbase==560?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HW240,hwbase==576?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HW250,hwbase==592?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HW260,hwbase==608?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HW280,hwbase==640?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_IRQ_3,irq==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_IRQ_5,irq==5?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_IRQ_7,irq==7?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_IRQ_9,irq==9?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_IRQ_10,irq==10?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_IRQ_11,irq==11?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_IRQ_12,irq==12?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_DMA_0,dma==0?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_DMA_1,dma==1?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_DMA_3,dma==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_DMA_5,dma==5?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_DMA_6,dma==6?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_DMA_7,dma==7?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HDMA_0,hdma==0?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HDMA_1,hdma==1?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HDMA_3,hdma==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HDMA_5,hdma==5?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HDMA_6,hdma==6?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_HDMA_7,hdma==7?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_AUTO,(oplmode=="auto")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_NONE,(oplmode=="none")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_CMS,(oplmode=="cms")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_OPL2,(oplmode=="opl2")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_DUALOPL2,(oplmode=="dualopl2")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_OPL3,(oplmode=="opl3")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_HARDWARE,(oplmode=="hardware")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_HARDWAREGB,(oplmode=="hardwaregb")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_EMU_DEFAULT,(oplemu=="default")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_EMU_COMPAT,(oplemu=="compat")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_EMU_FAST,(oplemu=="fast")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_49716,oplrate==49716?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_48000,oplrate==48000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_44100,oplrate==44100?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_32000,oplrate==32000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_22050,oplrate==22050?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_16000,oplrate==16000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_11025,oplrate==11025?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_SB_OPL_8000,oplrate==8000?MF_CHECKED:MF_STRING);
		}
		sec = static_cast<Section_prop *>(control->GetSection("glide"));
		if(sec) {
			const std::string glide = sec->Get_string("glide");
			CheckMenuItem(m_handle,ID_GLIDE_TRUE,(glide=="true")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GLIDE_EMU,(glide=="emu")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GLIDE_FALSE,(glide=="false")?MF_CHECKED:MF_STRING);
		}
		sec = static_cast<Section_prop *>(control->GetSection("ne2000"));
		if(sec) {
			CheckMenuItem(m_handle,ID_NE2000_SECTION,sec->Get_bool("ne2000")?MF_CHECKED:MF_STRING);
		}

		sec=static_cast<Section_prop *>(control->GetSection("gus"));
		if(sec) {
			const bool gus = sec->Get_bool("gus");
			const int gusrate = sec->Get_int("gusrate");
			const int gusbase = sec->Get_hex("gusbase");
			const int gusirq = sec->Get_int("gusirq");
			const int gusdma = sec->Get_int("gusdma");
			CheckMenuItem(m_handle,ID_GUS_TRUE,gus?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_FALSE,!gus?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_49716,gusrate==49716?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_48000,gusrate==48000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_44100,gusrate==44100?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_32000,gusrate==32000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_22050,gusrate==22050?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_16000,gusrate==16000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_11025,gusrate==11025?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_8000,gusrate==8000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_300,gusbase==768?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_280,gusbase==640?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_260,gusbase==608?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_240,gusbase==576?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_220,gusbase==544?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_2a0,gusbase==672?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_2c0,gusbase==704?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_2e0,gusbase==736?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_IRQ_3,gusirq==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_IRQ_5,gusirq==5?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_IRQ_7,gusirq==7?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_IRQ_9,gusirq==9?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_IRQ_10,gusirq==10?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_IRQ_11,gusirq==11?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_IRQ_12,gusirq==12?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_DMA_0,gusdma==0?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_DMA_1,gusdma==1?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_DMA_3,gusdma==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_DMA_5,gusdma==5?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_DMA_6,gusdma==6?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_GUS_DMA_7,gusdma==7?MF_CHECKED:MF_STRING);
		}
		sec=static_cast<Section_prop *>(control->GetSection("innova"));
		if(sec) {
			const bool innova = sec->Get_bool("innova");
			const int samplerate = sec->Get_int("samplerate");
			const int sidbase = sec->Get_hex("sidbase");
			const int quality = sec->Get_int("quality");
			CheckMenuItem(m_handle,ID_INNOVA_TRUE,innova?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_FALSE,!innova?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_49716,samplerate==49716?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_48000,samplerate==48000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_44100,samplerate==44100?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_32000,samplerate==32000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_22050,samplerate==22050?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_16000,samplerate==16000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_11025,samplerate==11025?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_8000,samplerate==8000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_280,sidbase==640?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_2A0,sidbase==672?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_2C0,sidbase==704?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_2E0,sidbase==736?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_220,sidbase==544?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_240,sidbase==576?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_260,sidbase==608?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_300,sidbase==768?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_3,quality==3?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_2,quality==2?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_1,quality==1?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_INNOVA_0,quality==0?MF_CHECKED:MF_STRING);
		}
		sec=static_cast<Section_prop *>(control->GetSection("speaker"));
		if(sec) {
			const bool pcspeaker = sec->Get_bool("pcspeaker");
			const int pcrate = sec->Get_int("pcrate");
			const std::string tandy = sec->Get_string("tandy");
			const int tandyrate = sec->Get_int("tandyrate");
			const bool disney = sec->Get_bool("disney");
			const std::string ps1audio = sec->Get_string("ps1audio");
			const int ps1audiorate = sec->Get_int("ps1audiorate");
			CheckMenuItem(m_handle,ID_PS1_ON,(ps1audio=="on")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_OFF,(ps1audio=="off")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_49716,ps1audiorate==49716?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_48000,ps1audiorate==48000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_44100,ps1audiorate==44100?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_32000,ps1audiorate==32000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_22050,ps1audiorate==22050?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_16000,ps1audiorate==16000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_11025,ps1audiorate==11025?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PS1_8000,ps1audiorate==8000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_TRUE,pcspeaker?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_FALSE,!pcspeaker?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_49716,pcrate==49716?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_48000,pcrate==48000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_44100,pcrate==44100?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_32000,pcrate==32000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_22050,pcrate==22050?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_16000,pcrate==16000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_11025,pcrate==11025?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_PCSPEAKER_8000,pcrate==8000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_ON,(tandy=="on")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_OFF,(tandy=="off")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_AUTO,(tandy=="auto")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_49716,tandyrate==49716?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_48000,tandyrate==48000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_44100,tandyrate==44100?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_32000,tandyrate==32000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_22050,tandyrate==22050?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_16000,tandyrate==16000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_11025,tandyrate==11025?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_TANDY_8000,tandyrate==8000?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_DISNEY_TRUE,disney?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_DISNEY_FALSE,!disney?MF_CHECKED:MF_STRING);
		}
		sec=static_cast<Section_prop *>(control->GetSection("render"));
		if(sec) {
			CheckMenuItem(m_handle,ID_LINEWISE,sec->Get_bool("linewise")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_CHAR9,sec->Get_bool("char9")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_MULTISCAN,sec->Get_bool("multiscan")?MF_CHECKED:MF_STRING);
		}
		sec=static_cast<Section_prop *>(control->GetSection("vsync"));
		if(sec) {
			const std::string vsyncmode = sec->Get_string("vsyncmode");
			EnableMenuItem(m_handle,ID_VSYNC_UP,((vsyncmode=="off") || (vsyncmode=="host"))?MF_GRAYED:MF_ENABLED);
			EnableMenuItem(m_handle,ID_VSYNC_DOWN,((vsyncmode=="off") || (vsyncmode=="host"))?MF_GRAYED:MF_ENABLED);
			CheckMenuItem(m_handle,ID_VSYNC_ON,(vsyncmode=="on")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_VSYNC_OFF,(vsyncmode=="off")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_VSYNC_HOST,(vsyncmode=="host")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_VSYNC_FORCE,(vsyncmode=="force")?MF_CHECKED:MF_STRING);
		}
		char* sdl_videodrv = getenv("SDL_VIDEODRIVER");
		CheckMenuItem(m_handle,ID_DRVFORCE_DIRECTX,((!strcmp(sdl_videodrv,"directx")) && (load_videodrv))?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_DRVFORCE_WINDIB,((!strcmp(sdl_videodrv,"windib")) && (load_videodrv))?MF_CHECKED:MF_STRING);
		CheckMenuItem(m_handle,ID_DRVFORCE_AUTO,!load_videodrv?MF_CHECKED:MF_STRING);
		extern bool Mouse_Vertical;
		CheckMenuItem(m_handle,ID_MOUSE_VERTICAL,Mouse_Vertical?MF_CHECKED:MF_STRING);
		sec=static_cast<Section_prop *>(control->GetSection("dos"));
		if(sec) {
			const std::string ems = sec->Get_string("ems");
			CheckMenuItem(m_handle,ID_XMS,sec->Get_bool("xms")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_EMS_TRUE,(ems=="true")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_EMS_FALSE,(ems=="false")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_EMS_EMSBOARD,(ems=="emsboard")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_EMS_EMM386,(ems=="emm386")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_UMB,sec->Get_bool("umb")?MF_CHECKED:MF_STRING);

			const std::string key = sec->Get_string("keyboardlayout");
			CheckMenuItem(m_handle,ID_KEY_NONE,(key=="auto")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_BG,(key=="bg")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_CZ,(key=="CZ")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_FR,(key=="fr")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_GK,(key=="gk")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_GR,(key=="gr")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_HR,(key=="hr")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_HU,(key=="hu")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_IT,(key=="it")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_NL,(key=="nl")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_NO,(key=="no")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_PL,(key=="pl")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_RU,(key=="ru")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_SK,(key=="sk")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_SP,(key=="sp")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_SU,(key=="su")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_SV,(key=="sv")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_BE,(key=="be")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_BR,(key=="br")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_CF,(key=="cf")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_DK,(key=="dk")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_LA,(key=="la")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_PO,(key=="po")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_SF,(key=="sf")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_SG,(key=="sg")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_UK,(key=="uk")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_US,(key=="us")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_YU,(key=="yu")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_FO,(key=="fo")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_MK,(key=="mk")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_MT,(key=="mt")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_PH,(key=="ph")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_RO,(key=="ro")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_SQ,(key=="sq")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_TM,(key=="tm")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_TR,(key=="tr")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_UX,(key=="ux")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_YC,(key=="yc")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_DV,(key=="dv")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_RH,(key=="rh")?MF_CHECKED:MF_STRING);
			CheckMenuItem(m_handle,ID_KEY_LH,(key=="lh")?MF_CHECKED:MF_STRING);
		}
		sec=static_cast<Section_prop *>(control->GetSection("ipx"));
		if(sec)	CheckMenuItem(m_handle,ID_IPXNET,sec->Get_bool("ipx")?MF_CHECKED:MF_STRING);

		sec=0;

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_A,(Drives['A'-'A'] || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_A,(Drives['A'-'A'] || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_A,(Drives['A'-'A'] || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_A,(Drives['A'-'A'] || menu.boot)?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_A,AUTOMOUNT("A:\\",'A') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_A,(!Drives['A'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_B,(Drives['B'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_B,(Drives['B'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_B,(Drives['B'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_B,(Drives['B'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_B,AUTOMOUNT("B:\\",'B') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_B,(!Drives['B'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_C,(Drives['C'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_C,(Drives['C'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_C,(Drives['C'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_C,(Drives['C'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_C,AUTOMOUNT("C:\\",'C') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_C,(!Drives['C'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_D,(Drives['D'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_D,(Drives['D'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_D,(Drives['D'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_D,(Drives['D'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_D,AUTOMOUNT("D:\\",'D') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_D,(!Drives['D'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_E,(Drives['E'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_E,(Drives['E'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_E,(Drives['E'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_E,(Drives['E'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_E,AUTOMOUNT("E:\\",'E') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_E,(!Drives['E'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_F,(Drives['F'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_F,(Drives['F'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_F,(Drives['F'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_F,(Drives['F'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_F,AUTOMOUNT("F:\\",'F') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_F,(!Drives['F'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_G,(Drives['G'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_G,(Drives['G'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_G,(Drives['G'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_G,(Drives['G'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_G,AUTOMOUNT("G:\\",'G') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_G,(!Drives['G'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_H,(Drives['H'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_H,(Drives['H'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_H,(Drives['H'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_H,(Drives['H'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_H,AUTOMOUNT("H:\\",'H') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_H,(!Drives['H'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_I,(Drives['I'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_I,(Drives['I'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_I,(Drives['I'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_I,(Drives['I'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_I,AUTOMOUNT("I:\\",'I') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_I,(!Drives['I'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_J,(Drives['J'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_J,(Drives['J'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_J,(Drives['J'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_J,(Drives['J'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_J,AUTOMOUNT("J:\\",'J') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_J,(!Drives['J'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_K,(Drives['K'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_K,(Drives['K'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_K,(Drives['K'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_K,(Drives['K'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_K,AUTOMOUNT("K:\\",'K') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_K,(!Drives['K'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_L,(Drives['L'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_L,(Drives['L'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_L,(Drives['L'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_L,(Drives['L'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_L,AUTOMOUNT("L:\\",'L') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_L,(!Drives['L'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_M,(Drives['M'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_M,(Drives['M'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_M,(Drives['M'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_M,(Drives['M'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_M,AUTOMOUNT("M:\\",'M') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_M,(!Drives['M'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_N,(Drives['N'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_N,(Drives['N'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_N,(Drives['N'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_N,(Drives['N'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_N,AUTOMOUNT("N:\\",'N') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_N,(!Drives['N'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_O,(Drives['O'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_O,(Drives['O'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_O,(Drives['O'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_O,(Drives['O'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_O,AUTOMOUNT("O:\\",'O') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_O,(!Drives['O'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_P,(Drives['P'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_P,(Drives['P'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_P,(Drives['P'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_P,(Drives['P'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_P,AUTOMOUNT("P:\\",'P') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_P,(!Drives['P'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_Q,(Drives['Q'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_Q,(Drives['Q'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_Q,(Drives['Q'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_Q,(Drives['Q'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_Q,AUTOMOUNT("Q:\\",'Q') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_Q,(!Drives['Q'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_R,(Drives['R'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_R,(Drives['R'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_R,(Drives['R'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_R,(Drives['R'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_R,AUTOMOUNT("R:\\",'R') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_R,(!Drives['R'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_S,(Drives['S'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_S,(Drives['S'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_S,(Drives['S'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_S,(Drives['S'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_S,AUTOMOUNT("S:\\",'S') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_S,(!Drives['S'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_T,(Drives['T'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_T,(Drives['T'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_T,(Drives['T'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_T,(Drives['T'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_T,AUTOMOUNT("T:\\",'T') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_T,(!Drives['T'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_U,(Drives['U'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_U,(Drives['U'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_U,(Drives['U'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_U,(Drives['U'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_U,AUTOMOUNT("U:\\",'U') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_U,(!Drives['U'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_V,(Drives['V'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_V,(Drives['V'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_V,(Drives['V'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_V,(Drives['V'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_V,AUTOMOUNT("V:\\",'V') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_V,(!Drives['V'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_W,(Drives['W'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_W,(Drives['W'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_W,(Drives['W'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_W,(Drives['W'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_W,AUTOMOUNT("W:\\",'W') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_W,(!Drives['W'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_X,(Drives['X'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_X,(Drives['X'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_X,(Drives['X'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_X,(Drives['X'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_X,AUTOMOUNT("X:\\",'X') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_X,(!Drives['X'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_Y,(Drives['Y'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_Y,(Drives['Y'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_Y,(Drives['Y'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_Y,(Drives['Y'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_Y,AUTOMOUNT("Y:\\",'Y') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_Y,(!Drives['Y'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);

		EnableMenuItem(m_handle,ID_MOUNT_CDROM_Z,(Drives['Z'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_FLOPPY_Z,(Drives['Z'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_LOCAL_Z,(Drives['Z'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_MOUNT_IMAGE_Z,(Drives['Z'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
		EnableMenuItem(m_handle,ID_AUTOMOUNT_Z,AUTOMOUNT("Z:\\",'Z') && !menu.boot?MF_ENABLED:MF_GRAYED);
		EnableMenuItem(m_handle,ID_UMOUNT_Z,(!Drives['Z'-'A']) || menu.boot?MF_GRAYED:MF_ENABLED);
}

void MSG_Loop(void) {
    if(!menu.gui || sdl.desktop.fullscreen) return;
    if (!GetMenu(GetHWND())) return;
    static MSG Message;
while (PeekMessage(&Message, GetHWND(), 0, 0, PM_REMOVE)) {
    switch (Message.message) {
	//case WM_SYSKEYDOWN:
	//case WM_SYSKEYUP:
	case WM_SYSCHAR:
	break;
        case WM_COMMAND:
                switch (LOWORD(Message.wParam)) {
		case ID_USESCANCODES: {
			Section* sec = control->GetSection("sdl");
			Section_prop * section=static_cast<Section_prop *>(sec); 
			SetVal("sdl","usescancodes",section->Get_bool("usescancodes")?"false":"true");
			}
			break;
		case ID_WAITONERR:
			if(sdl.wait_on_error) {
				SetVal("sdl","waitonerror","false");
				sdl.wait_on_error=false;
			} else {
				SetVal("sdl","waitonerror","true");
				sdl.wait_on_error=true;
			}
			break;
		case ID_HDD_SIZEUP:
			hdd_defsize+=3200;
			LOG_MSG("GUI: Current default freesize for local disk: %dMB",512*32*hdd_defsize/1024/1024);
			break;
		case ID_HDD_SIZEDOWN:
			hdd_defsize-=3200;
			LOG_MSG("GUI: Current default freesize for local disk: %dMB",512*32*hdd_defsize/1024/1024);
			break;
		case ID_BOOT_A:
			Go_Boot("A"); break;
		case ID_BOOT_C:
			Go_Boot("C"); break;
		case ID_BOOT_D:
			Go_Boot("D"); break;
		case ID_RESTART: {
            void restart_program(std::vector<std::string> & parameters);
            restart_program(control->startup_params);
            break;
        }
		case ID_QUIT: {
            throw(0);
        }
			break;
		case ID_OPENFILE:
			OpenFileDialog(0);
			break;
		case ID_PAUSE: {
			PauseDOSBox(1);
			break;
		}
		case ID_SAVESTATE_1: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(0); SaveGameState_Run(); break; }
		case ID_SAVESTATE_2: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(1); SaveGameState_Run(); break; }
		case ID_SAVESTATE_3: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(2); SaveGameState_Run(); break; }
		case ID_SAVESTATE_4: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(3); SaveGameState_Run(); break; }
		case ID_SAVESTATE_5: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(4); SaveGameState_Run(); break; }
		case ID_SAVESTATE_6: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(5); SaveGameState_Run(); break; }
		case ID_SAVESTATE_7: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(6); SaveGameState_Run(); break; }
		case ID_SAVESTATE_8: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(7); SaveGameState_Run(); break; }
		case ID_SAVESTATE_9: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(8); SaveGameState_Run(); break; }
		case ID_SAVESTATE_10: { extern void SaveGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(9); SaveGameState_Run(); break; }
		case ID_LOADSTATE_1: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(0); LoadGameState_Run(); break; }
		case ID_LOADSTATE_2: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(1); LoadGameState_Run(); break; }
		case ID_LOADSTATE_3: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(2); LoadGameState_Run(); break; }
		case ID_LOADSTATE_4: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(3); LoadGameState_Run(); break; }
		case ID_LOADSTATE_5: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(4); LoadGameState_Run(); break; }
		case ID_LOADSTATE_6: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(5); LoadGameState_Run(); break; }
		case ID_LOADSTATE_7: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(6); LoadGameState_Run(); break; }
		case ID_LOADSTATE_8: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(7); LoadGameState_Run(); break; }
		case ID_LOADSTATE_9: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(8); LoadGameState_Run(); break; }
		case ID_LOADSTATE_10: { extern void LoadGameState_Run(void); extern void SetGameState_Run(int value); SetGameState_Run(9); LoadGameState_Run(); break; }
		case ID_REMOVE_STATE_1: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/1/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/1/";
			}
			}

          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_2: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/2/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/2/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_3: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/3/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/3/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_4: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/4/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/4/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_5: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/5/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/5/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_6: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/6/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/6/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_7: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/7/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/7/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_8: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/8/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/8/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_9: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/9/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/9/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_10: {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/10/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/10/";
			}
			}
          struct dirent *pent;
        pdir=opendir(save_dir.c_str());

            while ((pent=readdir(pdir))) {
              std::string file_delete=pent->d_name;
              file_delete= save_dir.c_str()+file_delete;
              DeleteFile(file_delete.c_str());
             }
            closedir(pdir);
            break;
        }
		case ID_REMOVE_STATE_ALL: {
            for (int i=1; i<=10; i++) {
			DIR *pdir;
			std::string save_dir;
			struct stat status;
			bool Get_Custom_SaveDir(std::string& savedir);
			if(Get_Custom_SaveDir(save_dir)) {
				save_dir+="\\";
			} else {
			// check if save directory exists in current directory
			stat( "save", &status );
			if ( status.st_mode & S_IFDIR ) {
			// path exists in current directory
			save_dir="./save/";
			} else {
			// makes save to appdata
			Cross::GetPlatformConfigDir(save_dir);
			save_dir += "save/";
			}
			}
			std::string temp = save_dir;
			std::string s;
			std::stringstream out;
			out << i;
			s = out.str();
			temp += s;
			temp += "/";

                struct dirent *pent;
                pdir=opendir(temp.c_str());

                while ((pent=readdir(pdir))) {
                  std::string file_delete=pent->d_name;
                  file_delete= temp.c_str()+file_delete;
                  DeleteFile(file_delete.c_str());
                }
                closedir(pdir);
             }
            break;
        }
		case ID_NORMAL:
			if(strcasecmp(core_mode, "normal")==0) break;
			SetVal("cpu","core","normal");
			break;
#if (C_DYNAMIC_X86)
		case ID_DYNAMIC:
			if(strcasecmp(core_mode, "dynamic")==0) break;
			SetVal("cpu","core","dynamic");
			break;
#endif
		case ID_FULL:
			if(strcasecmp(core_mode, "full")==0) break;
			SetVal("cpu","core","full");
			break;
		case ID_SIMPLE:
			if(strcasecmp(core_mode, "simple")==0) break;
			SetVal("cpu","core","simple");
			break;
		case ID_AUTO:
			if(strcasecmp(core_mode, "auto")==0) break;
			SetVal("cpu","core","auto");
			break;
		case ID_KEYMAP: MAPPER_RunInternal(); break;
		case ID_AUTOCYCLE:
			CPU_CycleAutoAdjust = ! CPU_CycleAutoAdjust;
			GFX_SetTitle(CPU_CycleMax,-1,-1,false);
			break;
		case ID_AUTODETER:
			CPU_AutoDetermineMode = ! CPU_AutoDetermineMode;
			GFX_SetTitle(CPU_CycleMax,-1,-1,false);
			break;
		case ID_CAPMOUSE: GFX_CaptureMouse(); break;
		case ID_REFRESH: {
			extern void UI_Shortcut(int select); UI_Shortcut(1);
		}
		break;
		case ID_FULLSCREEN: GFX_SwitchFullScreen(); break;
		case ID_ASPECT:
			SetVal("render","aspect",render.aspect?"false":"true");
			break;
		case ID_HIDECYCL:
			menu.hidecycles = ! menu.hidecycles;
			GFX_SetTitle(CPU_CycleMax,-1,-1,false);
			break;
		case ID_TOGGLE:
			ToggleMenu(true);
			break;
		case ID_NONE: SCALER(scalerOpNormal,1) break; SetVal("render","scaler",!render.scale.forced?"none":"none forced"); break;
		case ID_NORMAL2X: SCALER(scalerOpNormal,2) break; SetVal("render","scaler",!render.scale.forced?"normal2x":"normal2x forced"); break;
		case ID_NORMAL3X: SCALER(scalerOpNormal,3) break; SetVal("render","scaler",!render.scale.forced?"normal3x":"normal3x forced"); break;
		case ID_HARDWARE2X: SCALER(scalerOpNormal,4) break; SetVal("render","scaler",!render.scale.forced?"hardware2x":"hardware2x forced"); break;
		case ID_HARDWARE3X: SCALER(scalerOpNormal,6) break; SetVal("render","scaler",!render.scale.forced?"hardware3x":"hardware3x forced"); break;
		case ID_ADVMAME2X: SCALER(scalerOpAdvMame,2) break; SetVal("render","scaler",!render.scale.forced?"advmame2x":"advmame2x forced"); break;
		case ID_ADVMAME3X: SCALER(scalerOpAdvMame,3) break; SetVal("render","scaler",!render.scale.forced?"advmame3x":"advmame3x forced"); break;
		case ID_ADVINTERP2X: SCALER(scalerOpAdvInterp,2) break; SetVal("render","scaler",!render.scale.forced?"advinterp2x":"advinterp2x forced"); break;
		case ID_ADVINTERP3X: SCALER(scalerOpAdvInterp,3) break; SetVal("render","scaler",!render.scale.forced?"advinterp3x":"advinterp3x forced"); break;
		case ID_HQ2X: SCALER(scalerOpHQ,2) break; SetVal("render","scaler",!render.scale.forced?"hq2x":"hq2x forced"); break;
		case ID_HQ3X: SCALER(scalerOpHQ,3) break; SetVal("render","scaler",!render.scale.forced?"hq3x":"hq3x forced"); break;
		case ID_2XSAI: SCALER(scalerOpSaI,2) break; SetVal("render","scaler",!render.scale.forced?"2xsai":"2xsai forced"); break;
		case ID_SUPER2XSAI: SCALER(scalerOpSuperSaI,2) break; SetVal("render","scaler",!render.scale.forced?"super2xsai":"super2xsai forced"); break;
		case ID_SUPEREAGLE: SCALER(scalerOpSuperEagle,2) break; SetVal("render","scaler",!render.scale.forced?"supereagle":"supereagle forced"); break;
		case ID_TV2X: SCALER(scalerOpTV,2) break; SetVal("render","scaler",!render.scale.forced?"tv2x":"tv2x forced"); break;
		case ID_TV3X: SCALER(scalerOpTV,3) break; SetVal("render","scaler",!render.scale.forced?"tv3x":"tv3x forced"); break;
		case ID_RGB2X: SCALER(scalerOpRGB,2) break; SetVal("render","scaler",!render.scale.forced?"rgb2x":"rgb2x forced"); break;
		case ID_RGB3X: SCALER(scalerOpRGB,3) break; SetVal("render","scaler",!render.scale.forced?"rgb3x":"rgb3x forced"); break;
		case ID_SCAN2X: SCALER(scalerOpScan,2) break; SetVal("render","scaler",!render.scale.forced?"scan2x":"scan2x forced"); break;
		case ID_SCAN3X: SCALER(scalerOpScan,3) break; SetVal("render","scaler",!render.scale.forced?"scan3x":"scan3x forced"); break;
		case ID_FORCESCALER: {
			SCALER(scalerOpNormal,1) SetVal("render","scaler",render.scale.forced?"none":"none forced"); else
			SCALER(scalerOpNormal,4) SetVal("render","scaler",render.scale.forced?"hardware2x":"hardware2x forced"); else
			SCALER(scalerOpNormal,6) SetVal("render","scaler",render.scale.forced?"hardware3x":"hardware3x forced"); else
			SCALER(scalerOpNormal,2) SetVal("render","scaler",render.scale.forced?"normal2x":"normal2x forced"); else
			SCALER(scalerOpNormal,3) SetVal("render","scaler",render.scale.forced?"normal3x":"normal3x forced"); else
			SCALER(scalerOpAdvMame,2) SetVal("render","scaler",render.scale.forced?"advmame2x":"advmame2x forced"); else
			SCALER(scalerOpAdvMame,3) SetVal("render","scaler",render.scale.forced?"advmame3x":"advmame3x forced"); else
			SCALER(scalerOpAdvInterp,2) SetVal("render","scaler",render.scale.forced?"advinterp2x":"advinterp2x forced"); else
			SCALER(scalerOpAdvInterp,3) SetVal("render","scaler",render.scale.forced?"advinterp3x":"advinterp3x forced"); else
			SCALER(scalerOpHQ,2) SetVal("render","scaler",render.scale.forced?"hq2x":"hq2x forced"); else
			SCALER(scalerOpHQ,3) SetVal("render","scaler",render.scale.forced?"hq3x":"hq3x forced"); else
			SCALER(scalerOpSaI,2) SetVal("render","scaler",render.scale.forced?"2xsai":"2xsai forced"); else
			SCALER(scalerOpSuperSaI,2) SetVal("render","scaler",render.scale.forced?"super2xsai":"super2xsai forced"); else
			SCALER(scalerOpSuperEagle,2) SetVal("render","scaler",render.scale.forced?"supereagle":"supereagle forced"); else
			SCALER(scalerOpTV,2) SetVal("render","scaler",render.scale.forced?"tv2x":"tv2x forced"); else
			SCALER(scalerOpTV,3) SetVal("render","scaler",render.scale.forced?"tv3x":"tv3x forced"); else
			SCALER(scalerOpRGB,2) SetVal("render","scaler",render.scale.forced?"rgb2x":"rgb2x forced"); else
			SCALER(scalerOpRGB,3) SetVal("render","scaler",render.scale.forced?"rgb3x":"rgb3x forced"); else
			SCALER(scalerOpScan,2) SetVal("render","scaler",render.scale.forced?"scan2x":"scan2x forced"); else
			SCALER(scalerOpScan,3) SetVal("render","scaler",render.scale.forced?"scan3x":"scan3x forced");
			break;
		}
		break;
		case ID_CYCLEUP:
            extern void CPU_CycleIncrease(bool pressed);
            CPU_CycleIncrease(1);
			 //cycle_conf();
			 break;
		case ID_CYCLEDOWN:
            extern void CPU_CycleDecrease(bool pressed);
            CPU_CycleDecrease(1);
			 //cycle_conf();
			 break;
		case ID_SKIPUP: {
			if (render.frameskip.max<10) render.frameskip.max++;
			LOG_MSG("Frame Skip at %d",(const char *)render.frameskip.max);
			char temp[10];
			sprintf(temp,"%d",render.frameskip.max);
			SetVal("render","frameskip",temp);
		}
			break;
		case ID_SKIPDOWN: {
			if (render.frameskip.max>0) render.frameskip.max--;
			LOG_MSG("Frame Skip at %d",render.frameskip.max);
			char temp[10];
			sprintf(temp,"%d",render.frameskip.max);
			SetVal("render","frameskip",temp);
		}
			break;
		case ID_UMOUNT_A: UnMount('A'); break;
		case ID_UMOUNT_B: UnMount('B'); break;
		case ID_UMOUNT_C: UnMount('C'); break;
		case ID_UMOUNT_D: UnMount('D'); break;
		case ID_UMOUNT_E: UnMount('E'); break;
		case ID_UMOUNT_F: UnMount('F'); break;
		case ID_UMOUNT_G: UnMount('G'); break;
		case ID_UMOUNT_H: UnMount('H'); break;
		case ID_UMOUNT_I: UnMount('I'); break;
		case ID_UMOUNT_J: UnMount('J'); break;
		case ID_UMOUNT_K: UnMount('K'); break;
		case ID_UMOUNT_L: UnMount('L'); break;
		case ID_UMOUNT_M: UnMount('M'); break;
		case ID_UMOUNT_N: UnMount('N'); break;
		case ID_UMOUNT_O: UnMount('O'); break;
		case ID_UMOUNT_P: UnMount('P'); break;
		case ID_UMOUNT_Q: UnMount('Q'); break;
		case ID_UMOUNT_R: UnMount('R'); break;
		case ID_UMOUNT_S: UnMount('S'); break;
		case ID_UMOUNT_T: UnMount('T'); break;
		case ID_UMOUNT_U: UnMount('U'); break;
		case ID_UMOUNT_V: UnMount('V'); break;
		case ID_UMOUNT_W: UnMount('W'); break;
		case ID_UMOUNT_X: UnMount('X'); break;
		case ID_UMOUNT_Y: UnMount('Y'); break;
		case ID_UMOUNT_Z: UnMount('Z'); break;
		case ID_AUTOMOUNT_A: MountDrive('A',"A:\\"); break;
		case ID_AUTOMOUNT_B: MountDrive('B',"B:\\"); break;
		case ID_AUTOMOUNT_C: MountDrive('C',"C:\\"); break;
		case ID_AUTOMOUNT_D: MountDrive('D',"D:\\"); break;
		case ID_AUTOMOUNT_E: MountDrive('E',"E:\\"); break;
		case ID_AUTOMOUNT_F: MountDrive('F',"F:\\"); break;
		case ID_AUTOMOUNT_G: MountDrive('G',"G:\\"); break;
		case ID_AUTOMOUNT_H: MountDrive('H',"H:\\"); break;
		case ID_AUTOMOUNT_I: MountDrive('I',"I:\\"); break;
		case ID_AUTOMOUNT_J: MountDrive('J',"J:\\"); break;
		case ID_AUTOMOUNT_K: MountDrive('K',"K:\\"); break;
		case ID_AUTOMOUNT_L: MountDrive('L',"L:\\"); break;
		case ID_AUTOMOUNT_M: MountDrive('M',"M:\\"); break;
		case ID_AUTOMOUNT_N: MountDrive('N',"N:\\"); break;
		case ID_AUTOMOUNT_O: MountDrive('O',"O:\\"); break;
		case ID_AUTOMOUNT_P: MountDrive('P',"P:\\"); break;
		case ID_AUTOMOUNT_Q: MountDrive('Q',"Q:\\"); break;
		case ID_AUTOMOUNT_R: MountDrive('R',"R:\\"); break;
		case ID_AUTOMOUNT_S: MountDrive('S',"S:\\"); break;
		case ID_AUTOMOUNT_T: MountDrive('T',"T:\\"); break;
		case ID_AUTOMOUNT_U: MountDrive('U',"U:\\"); break;
		case ID_AUTOMOUNT_V: MountDrive('V',"V:\\"); break;
		case ID_AUTOMOUNT_W: MountDrive('W',"W:\\"); break;
		case ID_AUTOMOUNT_X: MountDrive('X',"X:\\"); break;
		case ID_AUTOMOUNT_Y: MountDrive('Y',"Y:\\"); break;
		case ID_AUTOMOUNT_Z: MountDrive('Z',"Z:\\"); break;
		case ID_MOUNT_CDROM_A: BrowseFolder('A',"CDROM"); break;
		case ID_MOUNT_CDROM_B: BrowseFolder('B',"CDROM"); break;
		case ID_MOUNT_CDROM_C: BrowseFolder('C',"CDROM"); break;
		case ID_MOUNT_CDROM_D: BrowseFolder('D',"CDROM"); break;
		case ID_MOUNT_CDROM_E: BrowseFolder('E',"CDROM"); break;
		case ID_MOUNT_CDROM_F: BrowseFolder('F',"CDROM"); break;
		case ID_MOUNT_CDROM_G: BrowseFolder('G',"CDROM"); break;
		case ID_MOUNT_CDROM_H: BrowseFolder('H',"CDROM"); break;
		case ID_MOUNT_CDROM_I: BrowseFolder('I',"CDROM"); break;
		case ID_MOUNT_CDROM_J: BrowseFolder('J',"CDROM"); break;
		case ID_MOUNT_CDROM_K: BrowseFolder('K',"CDROM"); break;
		case ID_MOUNT_CDROM_L: BrowseFolder('L',"CDROM"); break;
		case ID_MOUNT_CDROM_M: BrowseFolder('M',"CDROM"); break;
		case ID_MOUNT_CDROM_N: BrowseFolder('N',"CDROM"); break;
		case ID_MOUNT_CDROM_O: BrowseFolder('O',"CDROM"); break;
		case ID_MOUNT_CDROM_P: BrowseFolder('P',"CDROM"); break;
		case ID_MOUNT_CDROM_Q: BrowseFolder('Q',"CDROM"); break;
		case ID_MOUNT_CDROM_R: BrowseFolder('R',"CDROM"); break;
		case ID_MOUNT_CDROM_S: BrowseFolder('S',"CDROM"); break;
		case ID_MOUNT_CDROM_T: BrowseFolder('T',"CDROM"); break;
		case ID_MOUNT_CDROM_U: BrowseFolder('U',"CDROM"); break;
		case ID_MOUNT_CDROM_V: BrowseFolder('V',"CDROM"); break;
		case ID_MOUNT_CDROM_W: BrowseFolder('W',"CDROM"); break;
		case ID_MOUNT_CDROM_X: BrowseFolder('X',"CDROM"); break;
		case ID_MOUNT_CDROM_Y: BrowseFolder('Y',"CDROM"); break;
		case ID_MOUNT_CDROM_Z: BrowseFolder('Z',"CDROM"); break;
		case ID_MOUNT_FLOPPY_A: BrowseFolder('A',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_B: BrowseFolder('B',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_C: BrowseFolder('C',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_D: BrowseFolder('D',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_E: BrowseFolder('E',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_F: BrowseFolder('F',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_G: BrowseFolder('G',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_H: BrowseFolder('H',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_I: BrowseFolder('I',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_J: BrowseFolder('J',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_K: BrowseFolder('K',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_L: BrowseFolder('L',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_M: BrowseFolder('M',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_N: BrowseFolder('N',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_O: BrowseFolder('O',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_P: BrowseFolder('P',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_Q: BrowseFolder('Q',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_R: BrowseFolder('R',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_S: BrowseFolder('S',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_T: BrowseFolder('T',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_U: BrowseFolder('U',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_V: BrowseFolder('V',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_W: BrowseFolder('W',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_X: BrowseFolder('X',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_Y: BrowseFolder('Y',"FLOPPY"); break;
		case ID_MOUNT_FLOPPY_Z: BrowseFolder('Z',"FLOPPY"); break;
		case ID_MOUNT_LOCAL_A: BrowseFolder('A',"LOCAL"); break;
		case ID_MOUNT_LOCAL_B: BrowseFolder('B',"LOCAL"); break;
		case ID_MOUNT_LOCAL_C: BrowseFolder('C',"LOCAL"); break;
		case ID_MOUNT_LOCAL_D: BrowseFolder('D',"LOCAL"); break;
		case ID_MOUNT_LOCAL_E: BrowseFolder('E',"LOCAL"); break;
		case ID_MOUNT_LOCAL_F: BrowseFolder('F',"LOCAL"); break;
		case ID_MOUNT_LOCAL_G: BrowseFolder('G',"LOCAL"); break;
		case ID_MOUNT_LOCAL_H: BrowseFolder('H',"LOCAL"); break;
		case ID_MOUNT_LOCAL_I: BrowseFolder('I',"LOCAL"); break;
		case ID_MOUNT_LOCAL_J: BrowseFolder('J',"LOCAL"); break;
		case ID_MOUNT_LOCAL_K: BrowseFolder('K',"LOCAL"); break;
		case ID_MOUNT_LOCAL_L: BrowseFolder('L',"LOCAL"); break;
		case ID_MOUNT_LOCAL_M: BrowseFolder('M',"LOCAL"); break;
		case ID_MOUNT_LOCAL_N: BrowseFolder('N',"LOCAL"); break;
		case ID_MOUNT_LOCAL_O: BrowseFolder('O',"LOCAL"); break;
		case ID_MOUNT_LOCAL_P: BrowseFolder('P',"LOCAL"); break;
		case ID_MOUNT_LOCAL_Q: BrowseFolder('Q',"LOCAL"); break;
		case ID_MOUNT_LOCAL_R: BrowseFolder('R',"LOCAL"); break;
		case ID_MOUNT_LOCAL_S: BrowseFolder('S',"LOCAL"); break;
		case ID_MOUNT_LOCAL_T: BrowseFolder('T',"LOCAL"); break;
		case ID_MOUNT_LOCAL_U: BrowseFolder('U',"LOCAL"); break;
		case ID_MOUNT_LOCAL_V: BrowseFolder('V',"LOCAL"); break;
		case ID_MOUNT_LOCAL_W: BrowseFolder('W',"LOCAL"); break;
		case ID_MOUNT_LOCAL_X: BrowseFolder('X',"LOCAL"); break;
		case ID_MOUNT_LOCAL_Y: BrowseFolder('Y',"LOCAL"); break;
		case ID_MOUNT_LOCAL_Z: BrowseFolder('Z',"LOCAL"); break;
		case ID_MOUNT_IMAGE_A: OpenFileDialog_Img('A'); break;
		case ID_MOUNT_IMAGE_B: OpenFileDialog_Img('B'); break;
		case ID_MOUNT_IMAGE_C: OpenFileDialog_Img('C'); break;
		case ID_MOUNT_IMAGE_D: OpenFileDialog_Img('D'); break;
		case ID_MOUNT_IMAGE_E: OpenFileDialog_Img('E'); break;
		case ID_MOUNT_IMAGE_F: OpenFileDialog_Img('F'); break;
		case ID_MOUNT_IMAGE_G: OpenFileDialog_Img('G'); break;
		case ID_MOUNT_IMAGE_H: OpenFileDialog_Img('H'); break;
		case ID_MOUNT_IMAGE_I: OpenFileDialog_Img('I'); break;
		case ID_MOUNT_IMAGE_J: OpenFileDialog_Img('J'); break;
		case ID_MOUNT_IMAGE_K: OpenFileDialog_Img('K'); break;
		case ID_MOUNT_IMAGE_L: OpenFileDialog_Img('L'); break;
		case ID_MOUNT_IMAGE_M: OpenFileDialog_Img('M'); break;
		case ID_MOUNT_IMAGE_N: OpenFileDialog_Img('N'); break;
		case ID_MOUNT_IMAGE_O: OpenFileDialog_Img('O'); break;
		case ID_MOUNT_IMAGE_P: OpenFileDialog_Img('P'); break;
		case ID_MOUNT_IMAGE_Q: OpenFileDialog_Img('Q'); break;
		case ID_MOUNT_IMAGE_R: OpenFileDialog_Img('R'); break;
		case ID_MOUNT_IMAGE_S: OpenFileDialog_Img('S'); break;
		case ID_MOUNT_IMAGE_T: OpenFileDialog_Img('T'); break;
		case ID_MOUNT_IMAGE_U: OpenFileDialog_Img('U'); break;
		case ID_MOUNT_IMAGE_V: OpenFileDialog_Img('V'); break;
		case ID_MOUNT_IMAGE_W: OpenFileDialog_Img('W'); break;
		case ID_MOUNT_IMAGE_X: OpenFileDialog_Img('X'); break;
		case ID_MOUNT_IMAGE_Y: OpenFileDialog_Img('Y'); break;
		case ID_MOUNT_IMAGE_Z: OpenFileDialog_Img('Z'); break;
		case ID_SSHOT: void CAPTURE_ScreenShotEvent(bool pressed); CAPTURE_ScreenShotEvent(true); break;
		case ID_MOVIE: void CAPTURE_VideoEvent(bool pressed); CAPTURE_VideoEvent(true); break;
		case ID_WAVE: void CAPTURE_WaveEvent(bool pressed); CAPTURE_WaveEvent(true); break;
		case ID_OPL: void OPL_SaveRawEvent(bool pressed); OPL_SaveRawEvent(true); break;
		case ID_MIDI: void CAPTURE_MidiEvent(bool pressed); CAPTURE_MidiEvent(true); break;
		case ID_XMS: mem_conf("xms", 0); break;
		case ID_EMS_TRUE: mem_conf("ems", 1); break;
		case ID_EMS_FALSE: mem_conf("ems", 2); break;
		case ID_EMS_EMSBOARD: mem_conf("ems", 3); break;
		case ID_EMS_EMM386: mem_conf("ems", 4); break;
		case ID_UMB: mem_conf("umb", 0); break;
		case ID_SEND_CTRL_ESC: {
			KEYBOARD_AddKey(KBD_leftctrl,true);
			KEYBOARD_AddKey(KBD_esc,true);
			KEYBOARD_AddKey(KBD_leftctrl,false);
			KEYBOARD_AddKey(KBD_esc,false);
			break;
		}
		case ID_SEND_ALT_TAB: {
			KEYBOARD_AddKey(KBD_leftalt,true);
			KEYBOARD_AddKey(KBD_tab,true);
			KEYBOARD_AddKey(KBD_leftalt,false);
			KEYBOARD_AddKey(KBD_tab,false);
			break;
		}
		case ID_SEND_CTRL_ALT_DEL: {
			KEYBOARD_AddKey(KBD_leftctrl,true);
			KEYBOARD_AddKey(KBD_leftalt,true);
			KEYBOARD_AddKey(KBD_delete,true);
			KEYBOARD_AddKey(KBD_leftctrl,false);
			KEYBOARD_AddKey(KBD_leftalt,false);
			KEYBOARD_AddKey(KBD_delete,false);
			break;
		}
        case ID_LINEWISE: {
            Section_prop * sec=0;
    		sec=static_cast<Section_prop *>(control->GetSection("render"));
            if(sec) {
                SetVal("render","linewise",!sec->Get_bool("linewise")?"true":"false");
            }
            break;
        }
        case ID_CHAR9: {
            Section_prop * sec=0;
    		sec=static_cast<Section_prop *>(control->GetSection("render"));
            if(sec) {
                SetVal("render","char9",!sec->Get_bool("char9")?"true":"false");
            }
            break;
        }
        case ID_MULTISCAN: {
            Section_prop * sec=0;
    		sec=static_cast<Section_prop *>(control->GetSection("render"));
            if(sec) {
                SetVal("render","multiscan",!sec->Get_bool("multiscan")?"true":"false");
            }
            break;
        }
		case ID_DRVFORCE_DIRECTX: {
				load_videodrv=true;
				putenv("SDL_VIDEODRIVER=directx");
				sdl.using_windib=false;
				void restart_program(std::vector<std::string> & parameters);
				restart_program(control->startup_params);
				break;
		}
		case ID_DRVFORCE_WINDIB: {
				load_videodrv=true;
				putenv("SDL_VIDEODRIVER=windib");
				sdl.using_windib=true;
				void restart_program(std::vector<std::string> & parameters);
				restart_program(control->startup_params);
				break;
		}
		case ID_DRVFORCE_AUTO: {
				load_videodrv=false;
				void restart_program(std::vector<std::string> & parameters);
				restart_program(control->startup_params);
				break;
		}
		case ID_VSYNC_ON: SetVal("vsync","vsyncmode","on"); break;
		case ID_VSYNC_HOST: SetVal("vsync","vsyncmode","host"); break;
		case ID_VSYNC_FORCE: SetVal("vsync","vsyncmode","force"); break;
		case ID_VSYNC_OFF: SetVal("vsync","vsyncmode","off"); break;
		case ID_SURFACE: if (sdl.desktop.want_type != SCREEN_SURFACE) { change_output(0); SetVal("sdl","output","surface"); } break;
		case ID_DDRAW: if (sdl.desktop.want_type != SCREEN_SURFACE_DDRAW) { change_output(1); SetVal("sdl","output","ddraw"); } break;
		case ID_OVERLAY: if (sdl.desktop.want_type != SCREEN_OVERLAY) { change_output(2); SetVal("sdl","output","overlay"); } break;
		case ID_OPENGL: change_output(3); SetVal("sdl","output","opengl"); break;
		case ID_OPENGLNB: change_output(4); SetVal("sdl","output","openglnb"); break;
		case ID_DIRECT3D: if (sdl.desktop.want_type != SCREEN_DIRECT3D) { change_output(5); SetVal("sdl","output","direct3d"); } break;
		case ID_OPENGLHQ: if (sdl.desktop.want_type != SCREEN_OPENGLHQ) {change_output(6); SetVal("sdl","output","openglhq"); } break;
		case ID_WINFULL_USER: case ID_WINRES_USER: extern void UI_Shortcut(int select); UI_Shortcut(2); break;
		case ID_WINRES_ORIGINAL: res_input(true,"original"); break;
		case ID_WINFULL_ORIGINAL: res_input(false,"original"); break;
		case ID_WINRES_DESKTOP: res_input(true,"desktop"); break;
		case ID_WINFULL_DESKTOP: res_input(false,"desktop"); break;
		case ID_WINRES_160x200: res_input(true,"160x200"); break;
		case ID_WINRES_320x200: res_input(true,"320x200"); break;
		case ID_WINRES_320x240: res_input(true,"320x240"); break;
		case ID_WINRES_640x200: res_input(true,"640x200"); break;
		case ID_WINRES_640x480: res_input(true,"640x480"); break;
		case ID_WINRES_720x350: res_input(true,"720x350"); break;
		case ID_WINRES_720x480: res_input(true,"720x480"); break;
		case ID_WINRES_720x576: res_input(true,"720x576"); break;
		case ID_WINRES_800x480: res_input(true,"800x480"); break;
		case ID_WINRES_800x600: res_input(true,"800x600"); break;
		case ID_WINRES_848x480: res_input(true,"848x480"); break;
		case ID_WINRES_852x480: res_input(true,"852x480"); break;
		case ID_WINRES_1024x512: res_input(true,"1024x512"); break;
		case ID_WINRES_1024x768: res_input(true,"1024x768"); break;
		case ID_WINRES_1152x864: res_input(true,"1152x864"); break;
		case ID_WINRES_1280x768: res_input(true,"1280x768"); break;
		case ID_WINRES_1280x800: res_input(true,"1280x800"); break;
		case ID_WINRES_1280x960: res_input(true,"1280x960"); break;
		case ID_WINRES_1280x1024: res_input(true,"1280x1024"); break;
		case ID_WINFULL_160x200: res_input(false,"160x200"); break;
		case ID_WINFULL_320x200: res_input(false,"320x200"); break;
		case ID_WINFULL_320x240: res_input(false,"320x240"); break;
		case ID_WINFULL_640x200: res_input(false,"640x200"); break;
		case ID_WINFULL_640x480: res_input(false,"640x480"); break;
		case ID_WINFULL_720x350: res_input(false,"720x350"); break;
		case ID_WINFULL_720x480: res_input(false,"720x480"); break;
		case ID_WINFULL_720x576: res_input(false,"720x576"); break;
		case ID_WINFULL_800x480: res_input(false,"800x480"); break;
		case ID_WINFULL_800x600: res_input(false,"800x600"); break;
		case ID_WINFULL_848x480: res_input(false,"848x480"); break;
		case ID_WINFULL_852x480: res_input(false,"852x480"); break;
		case ID_WINFULL_1024x512: res_input(false,"1024x512"); break;
		case ID_WINFULL_1024x768: res_input(false,"1024x768"); break;
		case ID_WINFULL_1152x864: res_input(false,"1152x864"); break;
		case ID_WINFULL_1280x768: res_input(false,"1280x768"); break;
		case ID_WINFULL_1280x800: res_input(false,"1280x800"); break;
		case ID_WINFULL_1280x960: res_input(false,"1280x960"); break;
		case ID_WINFULL_1280x1024: res_input(false,"1280x1024"); break;
		case ID_WINFULL_1360x765: res_input(false,"1360x765"); break;
		case ID_WINFULL_1366x768: res_input(false,"1366x768"); break;
		case ID_WINFULL_1400x1050: res_input(false,"1400x1050"); break;
		case ID_WINFULL_1680x1024: res_input(false,"1680x1024"); break;
		case ID_WINFULL_1680x1050: res_input(false,"1680x1050"); break;
		case ID_WINFULL_1600x1200: res_input(false,"1600x1200"); break;
		case ID_WINFULL_1792x1344: res_input(false,"1792x1344"); break;
		case ID_WINFULL_1800x1440: res_input(false,"1800x1440"); break;
		case ID_WINFULL_1856x1393: res_input(false,"1856x1393"); break;
		case ID_WINFULL_1920x1200: res_input(false,"1920x1200"); break;
		case ID_WINFULL_1920x1440: res_input(false,"1920x1440"); break;
		case ID_WINFULL_2048x1536: res_input(false,"2048x1536"); break;
		case ID_WINFULL_2560x1600: res_input(false,"2560x1600"); break;
		case ID_WINFULL_2560x2048: res_input(false,"2560x2048"); break;
		case ID_WINFULL_3200x2048: res_input(false,"3200x2048"); break;
		case ID_WINFULL_3200x2400: res_input(false,"3200x2400"); break;
		case ID_WINFULL_3840x2400: res_input(false,"3840x2400"); break;
		case ID_WINFULL_5120x4096: res_input(false,"5120x4096"); break;
		case ID_WINFULL_6400x4096: res_input(false,"6400x4096"); break;
		case ID_WINFULL_6400x4800: res_input(false,"6400x4800"); break;
		case ID_WINFULL_7680x4320: res_input(false,"7680x4320"); break;
		case ID_WINFULL_7680x4800: res_input(false,"7680x4800"); break;
		case ID_FULLDOUBLE:	{
			SetVal("sdl","fulldouble",sdl.desktop.doublebuf?"false":"true");
			res_init();
		}
			break;
		case ID_AUTOLOCK: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("sdl"));
			SetVal("sdl","autolock",sdl.mouse.autoenable?"false":"true");
			if(sec) sdl.mouse.autoenable=sec->Get_bool("autolock");
		}
			break;
		case ID_MOUSE:
			extern bool Mouse_Drv;
			Mouse_Drv =! Mouse_Drv;
			break;
		case ID_KEY_NONE: SetVal("dos","keyboardlayout","auto"); break;
		case ID_KEY_BG: SetVal("dos","keyboardlayout","bg"); break;
		case ID_KEY_CZ: SetVal("dos","keyboardlayout","cz"); break;
		case ID_KEY_FR: SetVal("dos","keyboardlayout","fr"); break;
		case ID_KEY_GK: SetVal("dos","keyboardlayout","gk"); break;
		case ID_KEY_GR: SetVal("dos","keyboardlayout","gr"); break;
		case ID_KEY_HR: SetVal("dos","keyboardlayout","hr"); break;
		case ID_KEY_HU: SetVal("dos","keyboardlayout","hu"); break;
		case ID_KEY_IT: SetVal("dos","keyboardlayout","it"); break;
		case ID_KEY_NL: SetVal("dos","keyboardlayout","nl"); break;
		case ID_KEY_NO: SetVal("dos","keyboardlayout","no"); break;
		case ID_KEY_PL: SetVal("dos","keyboardlayout","pl"); break;
		case ID_KEY_RU: SetVal("dos","keyboardlayout","ru"); break;
		case ID_KEY_SK: SetVal("dos","keyboardlayout","sk"); break;
		case ID_KEY_SP: SetVal("dos","keyboardlayout","sp"); break;
		case ID_KEY_SU: SetVal("dos","keyboardlayout","su"); break;
		case ID_KEY_SV: SetVal("dos","keyboardlayout","sv"); break;
		case ID_KEY_BE: SetVal("dos","keyboardlayout","be"); break;
		case ID_KEY_BR: SetVal("dos","keyboardlayout","br"); break;
		case ID_KEY_CF: SetVal("dos","keyboardlayout","cf"); break;
		case ID_KEY_DK: SetVal("dos","keyboardlayout","dk"); break;
		case ID_KEY_LA: SetVal("dos","keyboardlayout","la"); break;
		case ID_KEY_PO: SetVal("dos","keyboardlayout","po"); break;
		case ID_KEY_SF: SetVal("dos","keyboardlayout","sf"); break;
		case ID_KEY_SG: SetVal("dos","keyboardlayout","sg"); break;
		case ID_KEY_UK: SetVal("dos","keyboardlayout","uk"); break;
		case ID_KEY_US: SetVal("dos","keyboardlayout","us"); break;
		case ID_KEY_YU: SetVal("dos","keyboardlayout","yu"); break;
		case ID_KEY_FO: SetVal("dos","keyboardlayout","fo"); break;
		case ID_KEY_MK: SetVal("dos","keyboardlayout","mk"); break;
		case ID_KEY_MT: SetVal("dos","keyboardlayout","mt"); break;
		case ID_KEY_PH: SetVal("dos","keyboardlayout","ph"); break;
		case ID_KEY_RO: SetVal("dos","keyboardlayout","ro"); break;
		case ID_KEY_SQ: SetVal("dos","keyboardlayout","sq"); break;
		case ID_KEY_TM: SetVal("dos","keyboardlayout","tm"); break;
		case ID_KEY_TR: SetVal("dos","keyboardlayout","tr"); break;
		case ID_KEY_UX: SetVal("dos","keyboardlayout","ux"); break;
		case ID_KEY_YC: SetVal("dos","keyboardlayout","yc"); break;
		case ID_KEY_DV: SetVal("dos","keyboardlayout","dv"); break;
		case ID_KEY_RH: SetVal("dos","keyboardlayout","rh"); break;
		case ID_KEY_LH: SetVal("dos","keyboardlayout","lh"); break;
		case ID_MIDI_NONE: SetVal("midi","mpu401","none"); break;
		case ID_MIDI_UART: SetVal("midi","mpu401","uart"); break;
		case ID_MIDI_INTELLI: SetVal("midi","mpu401","intelligent"); break;
		case ID_MIDI_DEFAULT: SetVal("midi","mididevice","default"); break;
		case ID_MIDI_ALSA: SetVal("midi","mididevice","alsa"); break;
		case ID_MIDI_OSS: SetVal("midi","mididevice","oss"); break;
		case ID_MIDI_WIN32: SetVal("midi","mididevice","win32"); break;
		case ID_MIDI_COREAUDIO: SetVal("midi","mididevice","coreaudio"); break;
		case ID_MIDI_COREMIDI: SetVal("midi","mididevice","coremidi"); break;
		case ID_MIDI_MT32: SetVal("midi","mididevice","mt32"); break;
		case ID_MIDI_SYNTH: SetVal("midi","mididevice","synth"); break;
		case ID_MIDI_TIMIDITY: SetVal("midi","mididevice","timidity"); break;
		case ID_MIDI_MT32_REVERBMODE_AUTO: SetVal("midi","mt32.reverb.mode","auto"); break;
		case ID_MIDI_MT32_REVERBMODE_0: SetVal("midi","mt32.reverb.mode","0"); break;
		case ID_MIDI_MT32_REVERBMODE_1: SetVal("midi","mt32.reverb.mode","1"); break;
		case ID_MIDI_MT32_REVERBMODE_2: SetVal("midi","mt32.reverb.mode","2"); break;
		case ID_MIDI_MT32_REVERBMODE_3: SetVal("midi","mt32.reverb.mode","3"); break;
		case ID_MIDI_MT32_DAC_AUTO: SetVal("midi","mt32.dac","auto"); break;
		case ID_MIDI_MT32_DAC_0: SetVal("midi","mt32.dac","0"); break;
		case ID_MIDI_MT32_DAC_1: SetVal("midi","mt32.dac","1"); break;
		case ID_MIDI_MT32_DAC_2: SetVal("midi","mt32.dac","2"); break;
		case ID_MIDI_MT32_DAC_3: SetVal("midi","mt32.dac","3"); break;
		case ID_MIDI_MT32_REVERBTIME_0: SetVal("midi","mt32.reverb.time","0"); break;
		case ID_MIDI_MT32_REVERBTIME_1: SetVal("midi","mt32.reverb.time","1"); break;
		case ID_MIDI_MT32_REVERBTIME_2: SetVal("midi","mt32.reverb.time","2"); break;
		case ID_MIDI_MT32_REVERBTIME_3: SetVal("midi","mt32.reverb.time","3"); break;
		case ID_MIDI_MT32_REVERBTIME_4: SetVal("midi","mt32.reverb.time","4"); break;
		case ID_MIDI_MT32_REVERBTIME_5: SetVal("midi","mt32.reverb.time","5"); break;
		case ID_MIDI_MT32_REVERBTIME_6: SetVal("midi","mt32.reverb.time","6"); break;
		case ID_MIDI_MT32_REVERBTIME_7: SetVal("midi","mt32.reverb.time","7"); break;
		case ID_MIDI_MT32_REVERBLEV_0: SetVal("midi","mt32.reverb.level","0"); break;
		case ID_MIDI_MT32_REVERBLEV_1: SetVal("midi","mt32.reverb.level","1"); break;
		case ID_MIDI_MT32_REVERBLEV_2: SetVal("midi","mt32.reverb.level","2"); break;
		case ID_MIDI_MT32_REVERBLEV_3: SetVal("midi","mt32.reverb.level","3"); break;
		case ID_MIDI_MT32_REVERBLEV_4: SetVal("midi","mt32.reverb.level","4"); break;
		case ID_MIDI_MT32_REVERBLEV_5: SetVal("midi","mt32.reverb.level","5"); break;
		case ID_MIDI_MT32_REVERBLEV_6: SetVal("midi","mt32.reverb.level","6"); break;
		case ID_MIDI_MT32_REVERBLEV_7: SetVal("midi","mt32.reverb.level","7"); break;
		case ID_MIDI_MT32_REVERSESTEREO_TRUE: SetVal("midi","mt32ReverseStereo","on"); break;
		case ID_MIDI_MT32_REVERSESTEREO_FALSE: SetVal("midi","mt32ReverseStereo","off"); break;
		case ID_MIDI_DEV_NONE: SetVal("midi","mididevice","none"); break;
		case ID_GUS_TRUE: SetVal("gus","gus","true"); break;
		case ID_GUS_FALSE: SetVal("gus","gus","false"); break;
		case ID_GUS_49716: SetVal("gus","gusrate","49716"); break;
		case ID_GUS_48000: SetVal("gus","gusrate","48000"); break;
		case ID_GUS_44100: SetVal("gus","gusrate","44100"); break;
		case ID_GUS_32000:  SetVal("gus","gusrate","32000"); break;
		case ID_GUS_22050: SetVal("gus","gusrate","22050"); break;
		case ID_GUS_16000: SetVal("gus","gusrate","16000"); break;
		case ID_GUS_11025: SetVal("gus","gusrate","11025"); break;
		case ID_GUS_8000: SetVal("gus","gusrate","8000"); break;
		case ID_GUS_300: SetVal("gus","gusbase","300"); break;
		case ID_GUS_280: SetVal("gus","gusbase","280"); break;
		case ID_GUS_260: SetVal("gus","gusbase","260"); break;
		case ID_GUS_240: SetVal("gus","gusbase","240"); break;
		case ID_GUS_220: SetVal("gus","gusbase","220"); break;
		case ID_GUS_2a0: SetVal("gus","gusbase","2a0"); break;
		case ID_GUS_2c0: SetVal("gus","gusbase","2c0"); break;
		case ID_GUS_2e0: SetVal("gus","gusbase","2e0"); break;
		case ID_GUS_IRQ_3: SetVal("gus","gusirq","3"); break;
		case ID_GUS_IRQ_5: SetVal("gus","gusirq","5"); break;
		case ID_GUS_IRQ_7: SetVal("gus","gusirq","7"); break;
		case ID_GUS_IRQ_9: SetVal("gus","gusirq","9"); break;
		case ID_GUS_IRQ_10: SetVal("gus","gusirq","10"); break;
		case ID_GUS_IRQ_11: SetVal("gus","gusirq","11"); break;
		case ID_GUS_IRQ_12: SetVal("gus","gusirq","12"); break;
		case ID_GUS_DMA_0: SetVal("gus","gusdma","0"); break;
		case ID_GUS_DMA_1: SetVal("gus","gusdma","1"); break;
		case ID_GUS_DMA_3: SetVal("gus","gusdma","3"); break;
		case ID_GUS_DMA_5: SetVal("gus","gusdma","5"); break;
		case ID_GUS_DMA_6: SetVal("gus","gusdma","6"); break;
		case ID_GUS_DMA_7: SetVal("gus","gusdma","7"); break;
		case ID_INNOVA_TRUE: SetVal("innova","innova","true"); break;
		case ID_INNOVA_FALSE: SetVal("innova","innova","false"); break;
		case ID_INNOVA_49716: SetVal("innova","samplerate","49716"); break;
		case ID_INNOVA_48000: SetVal("innova","samplerate","48000"); break;
		case ID_INNOVA_44100: SetVal("innova","samplerate","44100"); break;
		case ID_INNOVA_32000: SetVal("innova","samplerate","32000"); break;
		case ID_INNOVA_22050: SetVal("innova","samplerate","22050"); break;
		case ID_INNOVA_11025: SetVal("innova","samplerate","11025"); break;
		case ID_INNOVA_8000: SetVal("innova","samplerate","8000"); break;
		case ID_INNOVA_280: SetVal("innova","sidbase","280"); break;
		case ID_INNOVA_2A0: SetVal("innova","sidbase","2a0"); break;
		case ID_INNOVA_2C0: SetVal("innova","sidbase","2c0"); break;
		case ID_INNOVA_2E0: SetVal("innova","sidbase","2e0"); break;
		case ID_INNOVA_220: SetVal("innova","sidbase","220"); break;
		case ID_INNOVA_240: SetVal("innova","sidbase","240"); break;
		case ID_INNOVA_260: SetVal("innova","sidbase","260"); break;
		case ID_INNOVA_300: SetVal("innova","sidbase","300"); break;
		case ID_INNOVA_3: SetVal("innova","quality","3"); break;
		case ID_INNOVA_2: SetVal("innova","quality","2"); break;
		case ID_INNOVA_1: SetVal("innova","quality","1"); break;
		case ID_INNOVA_0: SetVal("innova","quality","0"); break;
		case ID_PCSPEAKER_TRUE: SetVal("speaker","pcspeaker","true"); break;
		case ID_PCSPEAKER_FALSE: SetVal("speaker","pcspeaker","false"); break;
		case ID_PCSPEAKER_49716: SetVal("speaker","pcrate","49716"); break;
		case ID_PCSPEAKER_48000: SetVal("speaker","pcrate","48000"); break;
		case ID_PCSPEAKER_44100: SetVal("speaker","pcrate","44100"); break;
		case ID_PCSPEAKER_32000: SetVal("speaker","pcrate","32000"); break;
		case ID_PCSPEAKER_22050: SetVal("speaker","pcrate","22050"); break;
		case ID_PCSPEAKER_16000: SetVal("speaker","pcrate","16000"); break;
		case ID_PCSPEAKER_11025: SetVal("speaker","pcrate","11025"); break;
		case ID_PCSPEAKER_8000: SetVal("speaker","pcrate","8000"); break;
		case ID_PS1_ON: SetVal("speaker","ps1audio","on"); break;
		case ID_PS1_OFF: SetVal("speaker","ps1audio","off"); break;
		case ID_PS1_49716: SetVal("speaker","ps1audiorate","49716"); break;
		case ID_PS1_48000: SetVal("speaker","ps1audiorate","48000"); break;
		case ID_PS1_44100: SetVal("speaker","ps1audiorate","44100"); break;
		case ID_PS1_32000: SetVal("speaker","ps1audiorate","32000"); break;
		case ID_PS1_22050: SetVal("speaker","ps1audiorate","22050"); break;
		case ID_PS1_16000: SetVal("speaker","ps1audiorate","16000"); break;
		case ID_PS1_11025: SetVal("speaker","ps1audiorate","11025"); break;
		case ID_PS1_8000: SetVal("speaker","ps1audiorate","8000"); break;
		case ID_TANDY_ON: SetVal("speaker","tandy","on"); break;
		case ID_TANDY_OFF: SetVal("speaker","tandy","off"); break;
		case ID_TANDY_AUTO: SetVal("speaker","tandy","auto"); break;
		case ID_TANDY_49716: SetVal("speaker","tandyrate","49716"); break;
		case ID_TANDY_48000: SetVal("speaker","tandyrate","48000"); break;
		case ID_TANDY_44100: SetVal("speaker","tandyrate","44100"); break;
		case ID_TANDY_32000: SetVal("speaker","tandyrate","32000"); break;
		case ID_TANDY_22050: SetVal("speaker","tandyrate","22050"); break;
		case ID_TANDY_16000: SetVal("speaker","tandyrate","16000"); break;
		case ID_TANDY_11025: SetVal("speaker","tandyrate","11025"); break;
		case ID_TANDY_8000: SetVal("speaker","tandyrate","8000"); break;
		case ID_DISNEY_TRUE: SetVal("speaker","disney","true"); break;
		case ID_DISNEY_FALSE: SetVal("speaker","disney","false"); break;
		case ID_SB_NONE: SetVal("sblaster","sbtype","none"); break;
		case ID_SB_SB1: SetVal("sblaster","sbtype","sb1"); break;
		case ID_SB_SB2: SetVal("sblaster","sbtype","sb2"); break;
		case ID_SB_SBPRO1: SetVal("sblaster","sbtype","sbpro1"); break;
		case ID_SB_SBPRO2: SetVal("sblaster","sbtype","sbpro2"); break;
		case ID_SB_SB16: SetVal("sblaster","sbtype","sb16"); break;
		case ID_SB_SB16VIBRA: SetVal("sblaster","sbtype","sb16vibra"); break;
		case ID_SB_GB: SetVal("sblaster","sbtype","gb"); break;
		case ID_SB_300: SetVal("sblaster","sbbase","300"); break;
		case ID_SB_220: SetVal("sblaster","sbbase","220"); break;
		case ID_SB_240: SetVal("sblaster","sbbase","240"); break;
		case ID_SB_260: SetVal("sblaster","sbbase","260"); break;
		case ID_SB_280: SetVal("sblaster","sbbase","280"); break;
		case ID_SB_2a0: SetVal("sblaster","sbbase","2a0"); break;
		case ID_SB_2c0: SetVal("sblaster","sbbase","2c0"); break;
		case ID_SB_2e0: SetVal("sblaster","sbbase","2e0"); break;
		case ID_SB_HW210: SetVal("sblaster","hardwarebase","210"); break;
		case ID_SB_HW220: SetVal("sblaster","hardwarebase","220"); break;
		case ID_SB_HW230: SetVal("sblaster","hardwarebase","230"); break;
		case ID_SB_HW240: SetVal("sblaster","hardwarebase","240"); break;
		case ID_SB_HW250: SetVal("sblaster","hardwarebase","250"); break;
		case ID_SB_HW260: SetVal("sblaster","hardwarebase","260"); break;
		case ID_SB_HW280: SetVal("sblaster","hardwarebase","280"); break;
		case ID_SB_IRQ_3: SetVal("sblaster","irq","3"); break;
		case ID_SB_IRQ_5: SetVal("sblaster","irq","5"); break;
		case ID_SB_IRQ_7: SetVal("sblaster","irq","7"); break;
		case ID_SB_IRQ_9: SetVal("sblaster","irq","9"); break;
		case ID_SB_IRQ_10: SetVal("sblaster","irq","10"); break;
		case ID_SB_IRQ_11: SetVal("sblaster","irq","11"); break;
		case ID_SB_IRQ_12: SetVal("sblaster","irq","12"); break;
		case ID_SB_DMA_0: SetVal("sblaster","dma","0"); break;
		case ID_SB_DMA_1: SetVal("sblaster","dma","1"); break;
		case ID_SB_DMA_3: SetVal("sblaster","dma","3"); break;
		case ID_SB_DMA_5: SetVal("sblaster","dma","5"); break;
		case ID_SB_DMA_6: SetVal("sblaster","dma","6"); break;
		case ID_SB_DMA_7: SetVal("sblaster","dma","7"); break;
		case ID_SB_HDMA_0: SetVal("sblaster","hdma","0"); break;
		case ID_SB_HDMA_1: SetVal("sblaster","hdma","1"); break;
		case ID_SB_HDMA_3: SetVal("sblaster","hdma","3"); break;
		case ID_SB_HDMA_5: SetVal("sblaster","hdma","5"); break;
		case ID_SB_HDMA_6: SetVal("sblaster","hdma","6"); break;
		case ID_SB_HDMA_7: SetVal("sblaster","hdma","7"); break;
		case ID_SB_OPL_AUTO: SetVal("sblaster","oplmode","auto"); break;
		case ID_SB_OPL_NONE: SetVal("sblaster","oplmode","none"); break;
		case ID_SB_OPL_CMS: SetVal("sblaster","oplmode","cms"); break;
		case ID_SB_OPL_OPL2: SetVal("sblaster","oplmode","opl2"); break;
		case ID_SB_OPL_DUALOPL2: SetVal("sblaster","oplmode","dualopl2"); break;
		case ID_SB_OPL_OPL3: SetVal("sblaster","oplmode","opl3"); break;
		case ID_SB_OPL_HARDWARE: SetVal("sblaster","oplmode","hardware"); break;
		case ID_SB_OPL_HARDWAREGB: SetVal("sblaster","oplmode","hardwaregb"); break;
		case ID_SB_OPL_EMU_DEFAULT: SetVal("sblaster","oplemu","default"); break;
		case ID_SB_OPL_EMU_COMPAT: SetVal("sblaster","oplemu","compat"); break;
		case ID_SB_OPL_EMU_FAST: SetVal("sblaster","oplemu","fast"); break;
		case ID_SB_OPL_49716: SetVal("sblaster","oplrate","49716"); break;
		case ID_SB_OPL_48000: SetVal("sblaster","oplrate","48000"); break;
		case ID_SB_OPL_44100: SetVal("sblaster","oplrate","44100"); break;
		case ID_SB_OPL_32000: SetVal("sblaster","oplrate","32000"); break;
		case ID_SB_OPL_22050: SetVal("sblaster","oplrate","22050"); break;
		case ID_SB_OPL_16000: SetVal("sblaster","oplrate","16000"); break;
		case ID_SB_OPL_11025: SetVal("sblaster","oplrate","11025"); break;
		case ID_SB_OPL_8000: SetVal("sblaster","oplrate","8000"); break;
		case ID_OVERSCAN_0: LOG_MSG("GUI: Overscan 0 (surface)"); SetVal("sdl","overscan","0"); change_output(7); break;
		case ID_OVERSCAN_1: LOG_MSG("GUI: Overscan 1 (surface)"); SetVal("sdl","overscan","1"); change_output(7); break;
		case ID_OVERSCAN_2: LOG_MSG("GUI: Overscan 2 (surface)"); SetVal("sdl","overscan","2"); change_output(7); break;
		case ID_OVERSCAN_3: LOG_MSG("GUI: Overscan 3 (surface)"); SetVal("sdl","overscan","3"); change_output(7); break;
		case ID_OVERSCAN_4: LOG_MSG("GUI: Overscan 4 (surface)"); SetVal("sdl","overscan","4"); change_output(7); break;
		case ID_OVERSCAN_5: LOG_MSG("GUI: Overscan 5 (surface)"); SetVal("sdl","overscan","5"); change_output(7); break;
		case ID_OVERSCAN_6: LOG_MSG("GUI: Overscan 6 (surface)"); SetVal("sdl","overscan","6"); change_output(7); break;
		case ID_OVERSCAN_7: LOG_MSG("GUI: Overscan 7 (surface)"); SetVal("sdl","overscan","7"); change_output(7); break;
		case ID_OVERSCAN_8: LOG_MSG("GUI: Overscan 8 (surface)"); SetVal("sdl","overscan","8"); change_output(7); break;
		case ID_OVERSCAN_9: LOG_MSG("GUI: Overscan 9 (surface)"); SetVal("sdl","overscan","9"); change_output(7); break;
		case ID_OVERSCAN_10: LOG_MSG("GUI: Overscan 10 (surface)"); SetVal("sdl","overscan","10"); change_output(7); break;
		case ID_VSYNC_UP: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("vsync"));
			int vsyncrate = atoi(sec->Get_string("vsyncrate"))+5;
			char vsyncrate_str[10];
			itoa(vsyncrate, vsyncrate_str, 10);
			SetVal("vsync","vsyncrate",vsyncrate_str);
			LOG_MSG("GUI: Vertical Sync Rate set to %s",vsyncrate_str);
		}
			break;
		case ID_VSYNC_DOWN: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("vsync"));
			int vsyncrate = atoi(sec->Get_string("vsyncrate"))-5;
			char vsyncrate_str[10];
			itoa(vsyncrate, vsyncrate_str, 10);
			SetVal("vsync","vsyncrate",vsyncrate_str);
			LOG_MSG("GUI: Vertical Sync Rate set to %s",vsyncrate_str);
		}
			break;
		case ID_IPXNET: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("ipx"));
			const char * temp = sec->Get_bool("ipx")?"false":"true";
			SetVal("ipx","ipx",temp);
		}
			break;
		case ID_D3D_PS: D3D_PS(); if(sdl.desktop.want_type==SCREEN_DIRECT3D) change_output(7); break;
		case ID_JOYSTICKTYPE_AUTO: SetVal("joystick","joysticktype","auto"); break;
		case ID_JOYSTICKTYPE_2AXIS: SetVal("joystick","joysticktype","2axis"); break;
		case ID_JOYSTICKTYPE_4AXIS: SetVal("joystick","joysticktype","4axis"); break;
		case ID_JOYSTICKTYPE_4AXIS_2: SetVal("joystick","joysticktype","4axis_2"); break;
		case ID_JOYSTICKTYPE_FCS: SetVal("joystick","joysticktype","fcs"); break;
		case ID_JOYSTICKTYPE_CH: SetVal("joystick","joysticktype","ch"); break;
		case ID_JOYSTICKTYPE_NONE: SetVal("joystick","joysticktype","none"); break;
		case ID_JOYSTICK_TIMED: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("joystick"));
			const char * temp = sec->Get_bool("timed")?"false":"true";
			SetVal("joystick","timed",temp);
			break;
		}
        case ID_JOYSTICK_AUTOFIRE: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("joystick"));
			const char * temp = sec->Get_bool("autofire")?"false":"true";
			SetVal("joystick","autofire",temp);
            break;
        }
        case ID_JOYSTICK_SWAP34: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("joystick"));
			const char * temp = sec->Get_bool("swap34")?"false":"true";
			SetVal("joystick","swap34",temp);
            break;
        }
        case ID_JOYSTICK_BUTTONWRAP: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("joystick"));
			const char * temp = sec->Get_bool("buttonwrap")?"false":"true";
			SetVal("joystick","buttonwrap",temp);
            break;
        }
        /* case ID_DONGLE: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("parallel"));
			const char * temp = sec->Get_bool("dongle")?"false":"true";
			SetVal("parallel","dongle",temp); break;
		} */

		case ID_SWAPSTEREO: {
			Section_prop * sec=0;
			sec=static_cast<Section_prop *>(control->GetSection("mixer"));
			const char * temp = sec->Get_bool("swapstereo")?"false":"true";
			SetVal("mixer","swapstereo",temp); break;
			extern void MENU_swapstereo(bool enabled);
			MENU_swapstereo(sec->Get_bool("swapstereo"));
		}

        case ID_MUTE: {
		if(!menu.nosound) {
			//LOG_MSG("GUI: Mute");
			SDL_PauseAudio(1);
		} else {
			//LOG_MSG("GUI: Now playing");
			SDL_PauseAudio(0);
		}
		menu.nosound=!menu.nosound;
		break;
       }
        case ID_DOSBOX_SECTION: extern void UI_Shortcut(int select); UI_Shortcut(3); break;
        case ID_MIXER_SECTION: extern void UI_Shortcut(int select); UI_Shortcut(4); break;
        case ID_SERIAL_SECTION: extern void UI_Shortcut(int select); UI_Shortcut(5); break;
        case ID_PARALLEL_SECTION: extern void UI_Shortcut(int select); UI_Shortcut(11); break;
        case ID_PRINTER_SECTION: extern void UI_Shortcut(int select); UI_Shortcut(12); break;
        case ID_NE2000_SECTION: extern void UI_Shortcut(int select); UI_Shortcut(6); break;
        case ID_AUTOEXEC: extern void UI_Shortcut(int select); UI_Shortcut(7); break;
        case ID_MOUSE_VERTICAL: extern bool Mouse_Vertical; Mouse_Vertical =! Mouse_Vertical; break;
        case ID_GLIDE_TRUE: SetVal("glide","glide","true"); break;
        case ID_GLIDE_FALSE: SetVal("glide","glide","false"); break;
        case ID_GLIDE_EMU: SetVal("glide","glide","emu"); break;
        case ID_GLIDE_SECTION: extern void UI_Shortcut(int select); UI_Shortcut(8); break;
        case ID_SAVELANG: extern void UI_Shortcut(int select); UI_Shortcut(9); break;
        case ID_CPUTYPE_AUTO: SetVal("cpu","cputype","auto"); break;
        case ID_CPUTYPE_386: SetVal("cpu","cputype","386"); break;
        //case ID_CPUTYPE_386_SLOW: SetVal("cpu","cputype","386_slow"); break;
        case ID_CPUTYPE_386_PREFETCH: SetVal("cpu","cputype","386_prefetch"); break;
        case ID_CPUTYPE_486: SetVal("cpu","cputype","486"); break;
        case ID_CPUTYPE_PENTIUM: SetVal("cpu","cputype","pentium"); break;
        case ID_CPUTYPE_PENTIUM_MMX: SetVal("cpu","cputype","pentium_mmx"); break;
        case ID_CPU_ADVANCED: extern void UI_Shortcut(int select); UI_Shortcut(13); break;
        case ID_DOS_ADVANCED: extern void UI_Shortcut(int select); UI_Shortcut(14); break;
        case ID_MIDI_ADVANCED: extern void UI_Shortcut(int select); UI_Shortcut(15); break;
         default:
                 break;
                }
            default: {
            if(Message.message != WM_MOVE && Message.message != WM_NCHITTEST && Message.message != WM_MOUSEMOVE &&
            Message.message != WM_NCMOUSEMOVE && Message.message != WM_LBUTTONDOWN && Message.message != WM_RBUTTONDOWN &&
            Message.message != WM_MBUTTONDOWN && Message.message != WM_LBUTTONUP && Message.message != WM_RBUTTONUP &&
            Message.message != WM_MBUTTONUP && Message.message != WM_MOUSEWHEEL && Message.message != WM_MOVING && Message.message != WM_SIZING &&
            Message.message != WM_KEYUP && Message.message != WM_KEYDOWN && Message.message != WM_SYSKEYUP && Message.message != WM_SYSKEYDOWN &&
	Message.message != WM_ACTIVATE && Message.message != 0x0319 && Message.message != WM_HOTKEY && Message.message != WM_KILLFOCUS &&
	Message.message != WM_SYSDEADCHAR && Message.message != WM_CHAR)
            Reflect_Menu();

            if(!TranslateAccelerator(GetHWND(),0,&Message)) {
               TranslateMessage(&Message);
               DispatchMessage(&Message);
                  }
            } }
    }
}
#endif

void GFX_Events() {
	SDL_Event event;
#if defined (REDUCE_JOYSTICK_POLLING)
	static int poll_delay=0;
	int time=GetTicks();
	if (time-poll_delay>20) {
		poll_delay=time;
		if (sdl.num_joysticks>0) SDL_JoystickUpdate();
		MAPPER_UpdateJoysticks();
	}
#endif
	while (SDL_PollEvent(&event)) {
		switch (event.type) {
#ifdef __WIN32__
        case SDL_SYSWMEVENT : {
            if(menu_compatible) break;
            switch( event.syswm.msg->msg ) {
                case WM_SYSCOMMAND:
                    switch (event.syswm.msg->wParam) {
                        case SC_MAXIMIZE: case 0xF032: if(sdl.desktop.want_type==SCREEN_DIRECT3D) menu.maxwindow=true; else GFX_SwitchFullScreen();
                        break;
		     case 0xF122: case SC_RESTORE: menu.maxwindow=false; break;
						case SC_CLOSE:
							if(sdl.desktop.want_type==SCREEN_OPENGLHQ) {
								ShowWindow( GetConsoleWindow(), SW_HIDE );
								DestroyWindow(GetConsoleWindow());
								break;
							}
                }
                case WM_MOVE:
                      	break;
            	case WM_DROPFILES: {
            		char buff[50];
                    DragQueryFile((HDROP)event.syswm.msg->wParam,0,buff,200);
            		Drag_Drop(buff);
            		break;
           		}
                default:
        		break;
            }
        }
#endif
		case SDL_ACTIVEEVENT:
			if (event.active.state & SDL_APPINPUTFOCUS) {
				if (event.active.gain) {
					if (sdl.desktop.fullscreen && !sdl.mouse.locked)
						GFX_CaptureMouse();
					SetPriority(sdl.priority.focus);
					CPU_Disable_SkipAutoAdjust();
				} else {
					if (sdl.mouse.locked) {
#ifdef WIN32
						if (sdl.desktop.fullscreen) {
							VGA_KillDrawing();
							GFX_ForceFullscreenExit();
						}
#endif
						GFX_CaptureMouse();
					}
					SetPriority(sdl.priority.nofocus);
					GFX_LosingFocus();
					CPU_Enable_SkipAutoAdjust();
				}
			}

			/* Non-focus priority is set to pause; check to see if we've lost window or input focus
			 * i.e. has the window been minimised or made inactive?
			 */
			if (sdl.priority.nofocus == PRIORITY_LEVEL_PAUSE) {
				if ((event.active.state & (SDL_APPINPUTFOCUS | SDL_APPACTIVE)) && (!event.active.gain)) {
					/* Window has lost focus, pause the emulator.
					 * This is similar to what PauseDOSBox() does, but the exit criteria is different.
					 * Instead of waiting for the user to hit Alt-Break, we wait for the window to
					 * regain window or input focus.
					 */
					bool paused = true;
					SDL_Event ev;

					GFX_SetTitle(-1,-1,-1,true);
					KEYBOARD_ClrBuffer();
//					SDL_Delay(500);
//					while (SDL_PollEvent(&ev)) {
						// flush event queue.
//					}

					while (paused) {
						// WaitEvent waits for an event rather than polling, so CPU usage drops to zero
						SDL_WaitEvent(&ev);

						switch (ev.type) {
						case SDL_QUIT: throw(0); break; // a bit redundant at linux at least as the active events gets before the quit event.
						case SDL_ACTIVEEVENT:     // wait until we get window focus back
							if (ev.active.state & (SDL_APPINPUTFOCUS | SDL_APPACTIVE)) {
								// We've got focus back, so unpause and break out of the loop
								if (ev.active.gain) {
									paused = false;
									GFX_SetTitle(-1,-1,-1,false);
								}

								/* Now poke a "release ALT" command into the keyboard buffer
								 * we have to do this, otherwise ALT will 'stick' and cause
								 * problems with the app running in the DOSBox.
								 */
								KEYBOARD_AddKey(KBD_leftalt, false);
								KEYBOARD_AddKey(KBD_rightalt, false);
							}
							break;
						}
					}
				}
			}
			break;
		case SDL_MOUSEMOTION:
			HandleMouseMotion(&event.motion);
			break;
		case SDL_MOUSEBUTTONDOWN:
		case SDL_MOUSEBUTTONUP:
			HandleMouseButton(&event.button);
			break;
		case SDL_VIDEORESIZE:
    		HandleVideoResize(&event.resize);
			break;
		case SDL_QUIT:
			throw(0);
			break;
		case SDL_VIDEOEXPOSE:
			if ((sdl.draw.callback) && (!glide.enabled)) sdl.draw.callback( GFX_CallBackRedraw );
			break;
#ifdef WIN32
		case SDL_KEYDOWN:
		case SDL_KEYUP:
			// ignore event alt+tab
			if (event.key.keysym.sym==SDLK_LALT) sdl.laltstate = event.key.type;
			if (event.key.keysym.sym==SDLK_RALT) sdl.raltstate = event.key.type;
			if (((event.key.keysym.sym==SDLK_TAB)) &&
				//((sdl.laltstate==SDL_KEYDOWN) || (sdl.raltstate==SDL_KEYDOWN))) break;
				((sdl.laltstate==SDL_KEYDOWN) || (sdl.raltstate==SDL_KEYDOWN))) { MAPPER_LosingFocus(); break; }
#endif
#if defined (MACOSX)			
		case SDL_KEYDOWN:
		case SDL_KEYUP:
			/* On macs CMD-Q is the default key to close an application */
			if (event.key.keysym.sym == SDLK_q && (event.key.keysym.mod == KMOD_RMETA || event.key.keysym.mod == KMOD_LMETA) ) {
				KillSwitch(true);
				break;
			} 
#endif
		default:
			void MAPPER_CheckEvent(SDL_Event * event);
			MAPPER_CheckEvent(&event);
		}
	}
}

#if defined (WIN32)
static BOOL WINAPI ConsoleEventHandler(DWORD event) {
	switch (event) {
	case CTRL_SHUTDOWN_EVENT:
	case CTRL_LOGOFF_EVENT:
	case CTRL_CLOSE_EVENT:
	case CTRL_BREAK_EVENT:
		raise(SIGTERM);
		return TRUE;
	case CTRL_C_EVENT:
	default: //pass to the next handler
		return FALSE;
	}
}
#endif


/* static variable to show wether there is not a valid stdout.
 * Fixes some bugs when -noconsole is used in a read only directory */
bool no_stdout = false;
void GFX_ShowMsg(char const* format,...) {
	char buf[512];
	va_list msg;
	va_start(msg,format);
	vsprintf(buf,format,msg);
        strcat(buf,"\n");
	va_end(msg);
	if(!no_stdout) printf("%s",buf); //Else buf is parsed again.
}


void Config_Add_SDL() {
	Section_prop * sdl_sec=control->AddSection_prop("sdl",&GUI_StartUp);
	sdl_sec->AddInitFunction(&MAPPER_StartUp);
	Prop_bool* Pbool;
	Prop_string* Pstring;
	Prop_int* Pint;
	Prop_multival* Pmulti;

	Pbool = sdl_sec->Add_bool("fullscreen",Property::Changeable::Always,false);
	Pbool->Set_help("Start dosbox directly in fullscreen. (Press ALT-Enter to go back)");
     
	Pbool = sdl_sec->Add_bool("fulldouble",Property::Changeable::Always,false);
	Pbool->Set_help("Use double buffering in fullscreen. It can reduce screen flickering, but it can also result in a slow DOSBox.");

	//Pbool = sdl_sec->Add_bool("sdlresize",Property::Changeable::Always,false);
	//Pbool->Set_help("Makes window resizable (depends on scalers)");

	Pstring = sdl_sec->Add_string("fullresolution",Property::Changeable::Always,"desktop");
	Pstring->Set_help("What resolution to use for fullscreen: original or fixed size (e.g. 1024x768).\n"
	                  "  Using your monitor's native resolution with aspect=true might give the best results.\n"
			  "  If you end up with small window on a large screen, try an output different from surface.");

	Pstring = sdl_sec->Add_string("windowresolution",Property::Changeable::Always,"original");
	Pstring->Set_help("Scale the window to this size IF the output device supports hardware scaling.\n"
	                  "  (output=surface does not!)");

	const char* outputs[] = {
		"surface", "overlay",
#if C_OPENGL
		"opengl", "openglnb", "openglhq",
#endif
#if (HAVE_DDRAW_H) && defined(WIN32)
		"ddraw",
#endif
#if (HAVE_D3D9_H) && defined(WIN32)
		"direct3d",
#endif
		0 };
#ifdef __WIN32__
		Pstring = sdl_sec->Add_string("output",Property::Changeable::Always,"direct3d");
#else
		Pstring = sdl_sec->Add_string("output",Property::Changeable::Always,"surface");
#endif
	Pstring->Set_help("What video system to use for output.");
	Pstring->Set_values(outputs);

	Pbool = sdl_sec->Add_bool("autolock",Property::Changeable::Always,true);
	Pbool->Set_help("Mouse will automatically lock, if you click on the screen. (Press CTRL-F10 to unlock)");

	Pint = sdl_sec->Add_int("sensitivity",Property::Changeable::Always,100);
	Pint->SetMinMax(1,1000);
	Pint->Set_help("Mouse sensitivity.");

	Pbool = sdl_sec->Add_bool("waitonerror",Property::Changeable::Always, true);
	Pbool->Set_help("Wait before closing the console if dosbox has an error.");

	Pmulti = sdl_sec->Add_multi("priority", Property::Changeable::Always, ",");
	Pmulti->SetValue("higher,normal");
	Pmulti->Set_help("Priority levels for dosbox. Second entry behind the comma is for when dosbox is not focused/minimized.\n"
	                 "  pause is only valid for the second entry.");

	const char* actt[] = { "lowest", "lower", "normal", "higher", "highest", "pause", 0};
	Pstring = Pmulti->GetSection()->Add_string("active",Property::Changeable::Always,"higher");
	Pstring->Set_values(actt);

	const char* inactt[] = { "lowest", "lower", "normal", "higher", "highest", "pause", 0};
	Pstring = Pmulti->GetSection()->Add_string("inactive",Property::Changeable::Always,"normal");
	Pstring->Set_values(inactt);

	Pstring = sdl_sec->Add_path("mapperfile",Property::Changeable::Always,MAPPERFILE);
	Pstring->Set_help("File used to load/save the key/event mappings from. Resetmapper only works with the defaul value.");

#if (HAVE_D3D9_H) && (C_D3DSHADERS) && defined(WIN32)
	Pstring = sdl_sec->Add_string("pixelshader",Property::Changeable::Always,"none");
	Pstring->Set_help("Pixelshader program (effect file must be in Shaders subdirectory).");
#endif

	Pbool = sdl_sec->Add_bool("usescancodes",Property::Changeable::Always,false);
	Pbool->Set_help("Avoid usage of symkeys, might not work on all operating systems.");

	Pint = sdl_sec->Add_int("overscan",Property::Changeable::Always, 0);
	Pint->SetMinMax(0,10);
	Pint->Set_help("Width of overscan border (0 to 10). (works only if output=surface)");

//	Pint = sdl_sec->Add_int("overscancolor",Property::Changeable::Always, 0);
//	Pint->SetMinMax(0,1000);
//	Pint->Set_help("Value of overscan color.");
}

static void show_warning(char const * const message) {
	bool textonly = true;
#ifdef WIN32
	textonly = false;
	if ( !sdl.inited && SDL_Init(SDL_INIT_VIDEO|SDL_INIT_NOPARACHUTE) < 0 ) textonly = true;
	sdl.inited = true;
#endif
	fprintf(stderr, "Warning: %s", message);
	if(textonly) return;
	if(!sdl.surface) sdl.surface = SDL_SetVideoMode(640,400,0,SDL_RESIZABLE);
	if(!sdl.surface) return;
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
	Bit32u rmask = 0xff000000;
	Bit32u gmask = 0x00ff0000;
	Bit32u bmask = 0x0000ff00;
#else
	Bit32u rmask = 0x000000ff;
	Bit32u gmask = 0x0000ff00;                    
	Bit32u bmask = 0x00ff0000;
#endif
	SDL_Surface* splash_surf = SDL_CreateRGBSurface(SDL_SWSURFACE, 640, 400, 32, rmask, gmask, bmask, 0);
	if (!splash_surf) return;

	int x = 120,y = 20;
	std::string m(message),m2;
	std::string::size_type a,b,c,d;
   
	while(m.size()) { //Max 50 characters. break on space before or on a newline
		c = m.find('\n');
		d = m.rfind(' ',50);
		if(c>d) a=b=d; else a=b=c;
		if( a != std::string::npos) b++; 
		m2 = m.substr(0,a); m.erase(0,b);
		OutputString(x,y,m2.c_str(),0xffffffff,0,splash_surf);
		y += 20;
	}
   
	SDL_BlitSurface(splash_surf, NULL, sdl.surface, NULL);
	SDL_Flip(sdl.surface);
	SDL_Delay(12000);
}
   
static void launcheditor() {
	std::string path,file;
	Cross::CreatePlatformConfigDir(path);
	Cross::GetPlatformConfigName(file);
	path += file;
	FILE* f = fopen(path.c_str(),"r");
	if(!f && !control->PrintConfig(path.c_str())) {
		printf("tried creating %s. but failed.\n",path.c_str());
		exit(1);
	}
	if(f) fclose(f);
/*	if(edit.empty()) {
		printf("no editor specified.\n");
		exit(1);
	}*/
	std::string edit;
	while(control->cmdline->FindString("-editconf",edit,true)) //Loop until one succeeds
		execlp(edit.c_str(),edit.c_str(),path.c_str(),(char*) 0);
	//if you get here the launching failed!
	printf("can't find editor(s) specified at the command line.\n");
	exit(1);
}
#if C_DEBUG
extern void DEBUG_ShutDown(Section * /*sec*/);
#endif

void restart_program(std::vector<std::string> & parameters) {
	char** newargs = new char* [parameters.size()+1];
	// parameter 0 is the executable path
	// contents of the vector follow
	// last one is NULL
	for(Bitu i = 0; i < parameters.size(); i++) newargs[i]=(char*)parameters[i].c_str();
	newargs[parameters.size()] = NULL;
	if(sdl.desktop.fullscreen) SwitchFullScreen(1);
	if(!load_videodrv) putenv("SDL_VIDEODRIVER=");
#ifndef WIN32
	SDL_CloseAudio();
	SDL_Delay(50);
	SDL_Quit();
#if C_DEBUG
	// shutdown curses
	DEBUG_ShutDown(NULL);
#endif
#endif

#ifndef WIN32
	execvp(newargs[0], newargs);
#endif
#ifdef __MINGW32__
#ifdef WIN32 // if failed under win32
    PROCESS_INFORMATION pi;
    STARTUPINFO si; 
    ZeroMemory(&si,sizeof(si));
    si.cb=sizeof(si);
    ZeroMemory(&pi,sizeof(pi));

    if(CreateProcess(NULL, newargs[0], NULL, NULL, false, 0, NULL, NULL, &si, &pi)) {
		CloseHandle( pi.hProcess );
		CloseHandle( pi.hThread );
		SDL_CloseAudio();
		SDL_Delay(50);
        throw(0);
		SDL_Quit();
#if C_DEBUG
	// shutdown curses
		DEBUG_ShutDown(NULL);
#endif
    }
#endif
#else // if not MINGW
#ifdef WIN32
	char newargs_temp[32767];
	strcpy(newargs_temp, "");
	for(Bitu i = 1; i < parameters.size(); i++) {
		strcat(newargs_temp, " ");
		strcat(newargs_temp, newargs[i]);
	}

    if(ShellExecute(NULL, "open", newargs[0], newargs_temp, NULL, SW_SHOW)) {
		SDL_CloseAudio();
		SDL_Delay(50);
        throw(0);
		SDL_Quit();
#if C_DEBUG
	// shutdown curses
		DEBUG_ShutDown(NULL);
#endif
    }
#endif
#endif
	free(newargs);
}

void Restart(bool pressed) { // mapper handler
	restart_program(control->startup_params);
}

static void launchcaptures(std::string const& edit) {
	std::string path,file;
	Section* t = control->GetSection("dosbox");
	if(t) file = t->GetPropValue("captures");
	if(!t || file == NO_SUCH_PROPERTY) {
		printf("Config system messed up.\n");
		exit(1);
	}
	Cross::CreatePlatformConfigDir(path);
	path += file;
	Cross::CreateDir(path);
	struct stat cstat;
	if(stat(path.c_str(),&cstat) || (cstat.st_mode & S_IFDIR) == 0) {
		printf("%s doesn't exists or isn't a directory.\n",path.c_str());
		exit(1);
	}
/*	if(edit.empty()) {
		printf("no editor specified.\n");
		exit(1);
	}*/

	execlp(edit.c_str(),edit.c_str(),path.c_str(),(char*) 0);
	//if you get here the launching failed!
	printf("can't find filemanager %s\n",edit.c_str());
	exit(1);
}

static void printconfiglocation() {
	std::string path,file;
	Cross::CreatePlatformConfigDir(path);
	Cross::GetPlatformConfigName(file);
	path += file;
     
	FILE* f = fopen(path.c_str(),"r");
	if(!f && !control->PrintConfig(path.c_str())) {
		printf("tried creating %s. but failed",path.c_str());
		exit(1);
	}
	if(f) fclose(f);
	printf("%s\n",path.c_str());
	exit(0);
}

static void eraseconfigfile() {
	FILE* f = fopen("dosbox.conf","r");
	if(f) {
		fclose(f);
		show_warning("Warning: dosbox.conf exists in current working directory.\nThis will override the configuration file at runtime.\n");
	}
	std::string path,file;
	Cross::GetPlatformConfigDir(path);
	Cross::GetPlatformConfigName(file);
	path += file;
	f = fopen(path.c_str(),"r");
	if(!f) exit(0);
	fclose(f);
	unlink(path.c_str());
	exit(0);
}

static void erasemapperfile() {
	FILE* g = fopen("dosbox.conf","r");
	if(g) {
		fclose(g);
		show_warning("Warning: dosbox.conf exists in current working directory.\nKeymapping might not be properly reset.\n"
		             "Please reset configuration as well and delete the dosbox.conf.\n");
	}

	std::string path,file=MAPPERFILE;
	Cross::GetPlatformConfigDir(path);
	path += file;
	FILE* f = fopen(path.c_str(),"r");
	if(!f) exit(0);
	fclose(f);
	unlink(path.c_str());
	exit(0);
}

void SetNumLock( void ) {
#ifdef WIN32
	if (control->cmdline->FindExist("-disable_numlock_check")) return;
	//BYTE keyState[256];
	//GetKeyboardState((LPBYTE)&keyState);
      // Simulate a key press
         keybd_event( VK_NUMLOCK,
                      0x45,
                      KEYEVENTF_EXTENDEDKEY | 0,
                      0 );

      // Simulate a key release
         keybd_event( VK_NUMLOCK,
                      0x45,
                      KEYEVENTF_EXTENDEDKEY | KEYEVENTF_KEYUP,
                      0);
#endif
}

#if defined(WIN32) && !(C_DEBUG)
void DISP2_Init(Bit8u color);
#endif

int main(int argc, char* argv[]) {
	try {
		CommandLine com_line(argc,argv);
		Config myconf(&com_line);
		control=&myconf;
#ifdef WIN32
		BYTE keyState[256];
		GetKeyboardState((LPBYTE)&keyState);
		bool numlock_stat=false;
		if(keyState[VK_NUMLOCK] & 1) numlock_stat=true;
		if(numlock_stat) SetNumLock ();
#endif
		/* Init the configuration system and add default values */
		Config_Add_SDL();
		DOSBOX_Init();

		std::string editor;
		if(control->cmdline->FindString("-editconf",editor,false)) launcheditor();
		if(control->cmdline->FindString("-opencaptures",editor,true)) launchcaptures(editor);
		if(control->cmdline->FindExist("-eraseconf")) eraseconfigfile();
		if(control->cmdline->FindExist("-resetconf")) eraseconfigfile();
		if(control->cmdline->FindExist("-erasemapper")) erasemapperfile();
		if(control->cmdline->FindExist("-resetmapper")) erasemapperfile();
		
		/* Can't disable the console with debugger enabled */
#if defined(WIN32) && !(C_DEBUG)
		Bit8u disp2_color=0;
		std::string disp2_opt;
		if (control->cmdline->FindExist("-noconsole")) {
			ShowWindow( GetConsoleWindow(), SW_HIDE ); DestroyWindow(GetConsoleWindow());
			/* Redirect standard input and standard output */
			if (!control->cmdline->FindExist("-nolog")) {
			if(freopen(STDOUT_FILE, "w", stdout) == NULL)
				no_stdout = true; // No stdout so don't write messages
			freopen(STDERR_FILE, "w", stderr);
			setvbuf(stdout, NULL, _IOLBF, BUFSIZ);	/* Line buffered */
			setbuf(stderr, NULL);					/* No buffering */
			}
		} else if (control->cmdline->FindString("-display2",disp2_opt,false)) {
			if (strcasecmp(disp2_opt.c_str(),"amber")==0) disp2_color=1;
			else if (strcasecmp(disp2_opt.c_str(),"green")==0) disp2_color=2;
			DISP2_Init(disp2_color);
		} else {
			if (AllocConsole()) {
				if (!control->cmdline->FindExist("-nolog")) {
				fclose(stdin);
				fclose(stdout);
				fclose(stderr);
				freopen("CONIN$","r",stdin);
				freopen("CONOUT$","w",stdout);
				freopen("CONOUT$","w",stderr);
				}
			}
			SetConsoleTitle("DOSBox Status Window");
		}
#endif  //defined(WIN32) && !(C_DEBUG)
		if (control->cmdline->FindExist("-version") ||
		    control->cmdline->FindExist("--version") ) {
			printf("\nDOSBox version %s, copyright 2002-2011 DOSBox Team.\n\n",VERSION);
			printf("DOSBox is written by the DOSBox Team (See AUTHORS file))\n");
			printf("DOSBox comes with ABSOLUTELY NO WARRANTY.  This is free software,\n");
			printf("and you are welcome to redistribute it under certain conditions;\n");
			printf("please read the COPYING file thoroughly before doing so.\n\n");
			return 0;
		}
		if(control->cmdline->FindExist("-printconf")) printconfiglocation();

#if C_DEBUG
#if defined(WIN32)
	if (control->cmdline->FindExist("-noconsole")) {
		ShowWindow( GetConsoleWindow(), SW_HIDE );
		DestroyWindow(GetConsoleWindow());
	} else
#endif
		DEBUG_SetupConsole();
#endif

#if defined(WIN32)
	SetConsoleCtrlHandler((PHANDLER_ROUTINE) ConsoleEventHandler,TRUE);
#endif

#ifdef OS2
        PPIB pib;
        PTIB tib;
        DosGetInfoBlocks(&tib, &pib);
        if (pib->pib_ultype == 2) pib->pib_ultype = 3;
        setbuf(stdout, NULL);
        setbuf(stderr, NULL);
#endif

	/* Display Welcometext in the console */
	LOG_MSG("DOSBox version %s",VERSION);
	LOG_MSG("Copyright 2002-2011 DOSBox Team, published under GNU GPL.");

	int id, major, minor;
	DOSBox_CheckOS(id, major, minor);
	if (id==1) menu.compatible=true;
	if(!menu_compatible) {
	    if(DOSBox_Kor()) {
	        LOG_MSG("ڽ  ī http://cafe.daum.net/dosbox");
	        LOG_MSG("");
	} else
	LOG_MSG("---");
    }

	/* Init SDL */
#if SDL_VERSION_ATLEAST(1, 2, 14)
	/* Or debian/ubuntu with older libsdl version as they have done this themselves, but then differently.
	 * with this variable they will work correctly. I've only tested the 1.2.14 behaviour against the windows version
	 * of libsdl
	 */
	putenv(const_cast<char*>("SDL_DISABLE_LOCK_KEYS=1"));
#endif
#ifdef WIN32
	if (getenv("SDL_VIDEODRIVER")==NULL) {
		load_videodrv=false;
		putenv("SDL_VIDEODRIVER=windib");
		sdl.using_windib=true;
	}
#endif
	if ( SDL_Init( SDL_INIT_AUDIO|SDL_INIT_VIDEO|SDL_INIT_TIMER|SDL_INIT_CDROM
		|SDL_INIT_NOPARACHUTE
		) < 0 ) E_Exit("Can't init SDL %s",SDL_GetError());
	sdl.inited = true;

    if (control->cmdline->FindExist("-nogui") || menu.compatible) menu.gui=false;
    if (menu_gui && !control->cmdline->FindExist("-nomenu")) DOSBox_SetMenu();
    if (menu_gui) {
        if(GetMenu(GetHWND())) {
             if(!DOSBox_Kor())
                LOG_MSG("GUI: Press Ctrl-F10 to capture/release mouse.\n     Save your configuration and restart DOSBox if your settings do not take effect.");
            else {
                LOG_MSG("GUI: 콺  ϰų  Ϸ Ctrl-F10Ű ʽÿ.\n          ϰ DOSBox ٽ Ͻʽÿ.");
            }
       }
    } else {
	    LOG_MSG("GUI: Press Ctrl-F10 to capture/release mouse, Alt-F10 for configuration.");
    }
    SDL_Prepare();

#ifndef DISABLE_JOYSTICK
	//Initialise Joystick seperately. This way we can warn when it fails instead
	//of exiting the application
	if( SDL_InitSubSystem(SDL_INIT_JOYSTICK) < 0 ) LOG_MSG("Failed to init joystick support");
#endif

	sdl.laltstate = SDL_KEYUP;
	sdl.raltstate = SDL_KEYUP;

#if defined (WIN32)
#if SDL_VERSION_ATLEAST(1, 2, 10)
		sdl.using_windib=true;
#else
		sdl.using_windib=false;
#endif
		char sdl_drv_name[128];
		if (getenv("SDL_VIDEODRIVER")==NULL) {
			if (SDL_VideoDriverName(sdl_drv_name,128)!=NULL) {
				sdl.using_windib=false;
				if (strcmp(sdl_drv_name,"directx")!=0) {
					SDL_QuitSubSystem(SDL_INIT_VIDEO);
					putenv("SDL_VIDEODRIVER=directx");
					if (SDL_InitSubSystem(SDL_INIT_VIDEO)<0) {
						putenv("SDL_VIDEODRIVER=windib");
						if (SDL_InitSubSystem(SDL_INIT_VIDEO)<0) E_Exit("Can't init SDL Video %s",SDL_GetError());
						sdl.using_windib=true;
					}
				}
			}
		} else {
			char* sdl_videodrv = getenv("SDL_VIDEODRIVER");
			if (strcmp(sdl_videodrv,"directx")==0) sdl.using_windib = false;
			else if (strcmp(sdl_videodrv,"windib")==0) sdl.using_windib = true;
		}
#endif
	glide.fullscreen = &sdl.desktop.fullscreen;
	sdl.num_joysticks=SDL_NumJoysticks();

	/* Parse configuration files */
	std::string config_file,config_path;
	Cross::GetPlatformConfigDir(config_path);
	
	//First parse -userconf
	if(control->cmdline->FindExist("-userconf",true)){
		config_file.clear();
		Cross::GetPlatformConfigDir(config_path);
		Cross::GetPlatformConfigName(config_file);
		config_path += config_file;
		control->ParseConfigFile(config_path.c_str());
		if(!control->configfiles.size()) {
			//Try to create the userlevel configfile.
			config_file.clear();
			Cross::CreatePlatformConfigDir(config_path);
			Cross::GetPlatformConfigName(config_file);
			config_path += config_file;
			if(control->PrintConfig(config_path.c_str())) {
				LOG_MSG("CONFIG: Generating default configuration.\nWriting it to %s",config_path.c_str());
				//Load them as well. Makes relative paths much easier
				control->ParseConfigFile(config_path.c_str());
			}
		}
	}

	//Second parse -conf switches
	while(control->cmdline->FindString("-conf",config_file,true)) {
		if(!control->ParseConfigFile(config_file.c_str())) {
			// try to load it from the user directory
			control->ParseConfigFile((config_path + config_file).c_str());
		}
	}
	// if none found => parse localdir conf
	if(!control->configfiles.size()) control->ParseConfigFile("dosbox.conf");

	// if none found => parse userlevel conf
	if(!control->configfiles.size()) {
		config_file.clear();
		Cross::GetPlatformConfigName(config_file);
		control->ParseConfigFile((config_path + config_file).c_str());
	}

	if(!control->configfiles.size()) {
		//Try to create the userlevel configfile.
		config_file.clear();
		Cross::CreatePlatformConfigDir(config_path);
		Cross::GetPlatformConfigName(config_file);
		config_path += config_file;
		if(control->PrintConfig(config_path.c_str())) {
			LOG_MSG("CONFIG: Generating default configuration.\nWriting it to %s",config_path.c_str());
			//Load them as well. Makes relative paths much easier
			control->ParseConfigFile(config_path.c_str());
		} else {
			LOG_MSG("CONFIG: Using default settings. Create a configfile to change them");
		}
	}


#if (ENVIRON_LINKED)
		control->ParseEnv(environ);
#endif
		UI_Init();
		if (control->cmdline->FindExist("-startui")) UI_Run(false);
		/* Init all the sections */
		control->Init();

		/* Some extra SDL Functions */
		Section_prop * sdl_sec=static_cast<Section_prop *>(control->GetSection("sdl"));

		if (control->cmdline->FindExist("-fullscreen") || sdl_sec->Get_bool("fullscreen")) {
		if(sdl.desktop.want_type != SCREEN_OPENGLHQ) SetMenu(GetHWND(),NULL);
			if(!sdl.desktop.fullscreen) { //only switch if not already in fullscreen
				GFX_SwitchFullScreen();
			}
		}

		/* Init the keyMapper */
		MAPPER_Init();
		if (control->cmdline->FindExist("-startmapper")) MAPPER_RunInternal();
		/* Start up main machine */

		// Shows menu bar (window)
		//configfilesave = config_file;
		menu.startup=true;
		menu.hidecycles = ((control->cmdline->FindExist("-showcycles")) ? false : true);
        if(sdl.desktop.want_type==SCREEN_OPENGLHQ) {
            menu.gui=false; DOSBox_SetOriginalIcon();
            SetVal("render","scaler",!render.scale.forced?"hardware2x":"hardware2x forced");
        }

#ifdef WIN32
		if(sdl.desktop.want_type==SCREEN_OPENGL && sdl.using_windib) {
			SDL_QuitSubSystem(SDL_INIT_VIDEO);
			if (SDL_InitSubSystem(SDL_INIT_VIDEO)<0) E_Exit("Can't init SDL Video %s",SDL_GetError());
			if(sdl.opengl.bilinear) change_output(3); else change_output(4);
			GFX_SetIcon();
			SDL_Prepare();
			if (menu.gui && !control->cmdline->FindExist("-nomenu")) {
				SetMenu(GetHWND(), LoadMenu(GetModuleHandle(NULL),MAKEINTRESOURCE(IDR_MENU)));
				DrawMenuBar (GetHWND());
				//if (menu.gui && !control->cmdline->FindExist("-nomenu")) DOSBox_SetMenu();
			}
		}
#endif
		Section_prop *sec=static_cast<Section_prop *>(control->GetSection("sdl"));
#ifdef WIN32
		if(!strcmp(sec->Get_string("output"),"ddraw") && sdl.using_windib) {
			SDL_QuitSubSystem(SDL_INIT_VIDEO);
			putenv("SDL_VIDEODRIVER=directx");
			sdl.using_windib=false;
			if (SDL_InitSubSystem(SDL_INIT_VIDEO)<0) E_Exit("Can't init SDL Video %s",SDL_GetError());
			change_output(1);
			GFX_SetIcon();
			SDL_Prepare();
			if (menu.gui && !control->cmdline->FindExist("-nomenu")) {
				SetMenu(GetHWND(), LoadMenu(GetModuleHandle(NULL),MAKEINTRESOURCE(IDR_MENU)));
				DrawMenuBar (GetHWND());
				//if (menu.gui && !control->cmdline->FindExist("-nomenu")) DOSBox_SetMenu();
			}
		}
		if(!load_videodrv && numlock_stat) SetNumLock ();
#endif
		control->StartUp();
#ifdef __WIN32__
		//menu.startup=false;
#endif
		/* Shutdown everything */
	} catch (char * error) {
#if defined (WIN32)
		sticky_keys(true);
#endif
		GFX_ShowMsg("Exit to error: %s",error);
		fflush(NULL);
		if(sdl.wait_on_error) {
			//TODO Maybe look for some way to show message in linux?
#if (C_DEBUG)
			GFX_ShowMsg("Press enter to continue");
			fflush(NULL);
			fgetc(stdin);
#elif defined(WIN32)
			Sleep(5000);
#endif
		}

	}
	catch (int){
		;//nothing pressed killswitch
	}
	catch(...){
		sticky_keys(true);

		//Force visible mouse to end user. Somehow this sometimes doesn't happen
		SDL_WM_GrabInput(SDL_GRAB_OFF);
		SDL_ShowCursor(SDL_ENABLE);
		throw;//dunno what happened. rethrow for sdl to catch
	}

	sticky_keys(true); //Might not be needed if the shutdown function switches to windowed mode, but it doesn't hurt

	//Force visible mouse to end user. Somehow this sometimes doesn't happen
	SDL_WM_GrabInput(SDL_GRAB_OFF);
	SDL_ShowCursor(SDL_ENABLE);

	SDL_Quit();//Let's hope sdl will quit as well when it catches an exception
	return 0;
}

void GFX_GetSize(int &width, int &height, bool &fullscreen) {
	width = sdl.clip.w; // draw.width
	height = sdl.clip.h; // draw.height
	fullscreen = sdl.desktop.fullscreen;
}

void GFX_ShutDown(void) {
	GFX_Stop();
	if (sdl.draw.callback) (sdl.draw.callback)( GFX_CallBackStop );
	if (sdl.mouse.locked) GFX_CaptureMouse();
	if (sdl.desktop.fullscreen) GFX_SwitchFullScreen();
}

bool OpenGL_using(void) {
	return (sdl.desktop.want_type==SCREEN_OPENGL?true:false);
}

bool Get_Custom_SaveDir(std::string& savedir) {
	std::string custom_savedir;
	if (control->cmdline->FindString("-savedir",custom_savedir,false)) {
		savedir=custom_savedir;
		return true;
	} else {
		return false;
	}
}

// save state support
void POD_Save_Sdlmain( std::ostream& stream )
{
	// - pure data
	WRITE_POD( &sdl.mouse.autolock, sdl.mouse.autolock );
	WRITE_POD( &sdl.mouse.requestlock, sdl.mouse.requestlock );
}


void POD_Load_Sdlmain( std::istream& stream )
{
	// - pure data
	READ_POD( &sdl.mouse.autolock, sdl.mouse.autolock );
	READ_POD( &sdl.mouse.requestlock, sdl.mouse.requestlock );
}



/*
ykhwong svn-daum 2012-02-20

(misc excluded)


struct SDL_Block sdl
	// - system data
	bool inited;
	bool active;
	bool updating;

	// - struct system data
	struct {
		Bit32u width;
		Bit32u height;
		Bit32u bpp;
		Bitu flags;
		double scalex,scaley;
		GFX_CallBack_t callback;
	} draw;
	bool wait_on_error;

	// - struct system data
	struct {
		struct {
			Bit16u width, height;
			bool fixed;
		} full;
		struct {
			Bit16u width, height;
		} window;
		Bit8u bpp;
		bool fullscreen;
		bool lazy_fullscreen;
		bool lazy_fullscreen_req;
		bool doublebuf;
		SCREEN_TYPES type;
		SCREEN_TYPES want_type;
	} desktop;
#if C_OPENGL
	// - struct system data
	struct {
		Bitu pitch;
		void * framebuf;
		GLuint texture;
		GLuint displaylist;
		GLint max_texsize;
		bool bilinear;
		bool packed_pixel;
		bool paletted_texture;
#if defined(NVIDIA_PixelDataRange)
		bool pixel_data_range;
#endif
	} opengl;
#endif

	// - struct system data
	struct {
		SDL_Surface * surface;
#if (HAVE_DDRAW_H) && defined(WIN32)
		RECT rect;
#endif
	} blit;

	// - struct system data
	struct {
		PRIORITY_LEVELS focus;
		PRIORITY_LEVELS nofocus;
	} priority;
	SDL_Rect clip;
	SDL_Surface * surface;
	SDL_Overlay * overlay;
	SDL_cond *cond;

	// - near-system data
	struct {
		// - pure data (set by mouse driver)
		bool autolock;

		// - system data
		bool autoenable;

		// - pure data (set by mouse driver)
		bool requestlock;

		// - system data
		bool locked;
		Bitu sensitivity;
	} mouse;

	// - assume system data
	SDL_Rect updateRects[1024];
	Bitu overscan_color;
	Bitu overscan_width;
	Bitu num_joysticks;
#if defined (WIN32)
	bool using_windib;
#endif
	// state of alt-keys for certain special handlings
	Bit8u laltstate;
	Bit8u raltstate;
*/
