/*
 *  Portion of this software comes with the following license
 */

/***************************************************************************

    Copyright Aaron Giles
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions are
    met:

        * Redistributions of source code must retain the above copyright
          notice, this list of conditions and the following disclaimer.
        * Redistributions in binary form must reproduce the above copyright
          notice, this list of conditions and the following disclaimer in
          the documentation and/or other materials provided with the
          distribution.
        * Neither the name 'MAME' nor the names of its contributors may be
          used to endorse or promote products derived from this software
          without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY AARON GILES ''AS IS'' AND ANY EXPRESS OR
    IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
    DISCLAIMED. IN NO EVENT SHALL AARON GILES BE LIABLE FOR ANY DIRECT,
    INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
    (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
    SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
    STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
    IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
    POSSIBILITY OF SUCH DAMAGE.

***************************************************************************/

#ifndef VOODOO_MAIN_H
#define VOODOO_MAIN_H

/* enumeration specifying which model of Voodoo we are emulating */
enum
{
	VOODOO_1,
	VOODOO_2,
	VOODOO_BANSHEE,
	VOODOO_3,
	MAX_VOODOO_TYPES
};


#define STD_VOODOO_1_CLOCK			50000000
#define STD_VOODOO_2_CLOCK			90000000
#define STD_VOODOO_BANSHEE_CLOCK	90000000
#define STD_VOODOO_3_CLOCK			132000000


/***************************************************************************
    TYPE DEFINITIONS
***************************************************************************/

//typedef void (*voodoo_vblank_func)(running_device *device, int state);
//typedef void (*voodoo_stall_func)(running_device *device, int state);

typedef struct _voodoo_config voodoo_config;
struct _voodoo_config
{
	int					type;
	Bit8u				fbmem;
	Bit8u				tmumem0;
	Bit8u				tmumem1;
	const char *		screen;
	const char *		cputag;
//	voodoo_vblank_func	vblank;
//	voodoo_stall_func	stall;
};

#define VOODOO_MEM 0x60000000
#define VOODOO_REG_PAGES 1024
#define VOODOO_LFB_PAGES 1024
#define VOODOO_TEX_PAGES 2048
#define VOODOO_PAGES (VOODOO_REG_PAGES+VOODOO_LFB_PAGES+VOODOO_TEX_PAGES)

class  Voodoo_PageHandler;
extern Voodoo_PageHandler * voodoo_pagehandler;

void Voodoo_UpdateScreenStart();
void Voodoo_Output_Enable(bool enabled);
bool Voodoo_get_retrace();


#include <math.h>

/***************************************************************************
    TYPE DEFINITIONS
***************************************************************************/

/* core components of the attotime structure */
typedef Bit64s attoseconds_t;
typedef Bit32s seconds_t;


/* the attotime structure itself */
typedef struct _attotime attotime;
struct _attotime
{
	seconds_t		seconds;
	attoseconds_t	attoseconds;
};

#define ATTOSECONDS_PER_SECOND_SQRT		((attoseconds_t)1000000000)
#define ATTOSECONDS_PER_SECOND			(ATTOSECONDS_PER_SECOND_SQRT * ATTOSECONDS_PER_SECOND_SQRT)

/* convert between hertz (as a double) and attoseconds */
#define ATTOSECONDS_TO_HZ(x)			((double)ATTOSECONDS_PER_SECOND / (double)(x))
#define HZ_TO_ATTOSECONDS(x)			((attoseconds_t)(ATTOSECONDS_PER_SECOND / (x)))


typedef Bit32u offs_t;

/* poly_param_extent describes information for a single parameter in an extent */
typedef struct _poly_param_extent poly_param_extent;
struct _poly_param_extent
{
	float		start;						/* parameter value at starting X,Y */
	float		dpdx;						/* dp/dx relative to starting X */
};


#define MAX_VERTEX_PARAMS					6
/* poly_extent describes start/end points for a scanline, along with per-scanline parameters */
typedef struct _poly_extent poly_extent;
struct _poly_extent
{
	Bit16s		startx;						/* starting X coordinate (inclusive) */
	Bit16s		stopx;						/* ending X coordinate (exclusive) */
	poly_param_extent param[MAX_VERTEX_PARAMS];	/* starting and dx values for each parameter */
};

#define U64(x) (x)
#ifndef TRUE
#define TRUE true
#define FALSE false
#endif


/* an rgb_t is a single combined R,G,B (and optionally alpha) value */
typedef Bit32u rgb_t;

/* an rgb15_t is a single combined 15-bit R,G,B value */
typedef Bit16u rgb15_t;

/* macros to assemble rgb_t values */
#define MAKE_ARGB(a,r,g,b)	((((rgb_t)(a) & 0xff) << 24) | (((rgb_t)(r) & 0xff) << 16) | (((rgb_t)(g) & 0xff) << 8) | ((rgb_t)(b) & 0xff))
#define MAKE_RGB(r,g,b)		(MAKE_ARGB(255,r,g,b))

/* macros to extract components from rgb_t values */
#define RGB_ALPHA(rgb)		(((rgb) >> 24) & 0xff)
#define RGB_RED(rgb)		(((rgb) >> 16) & 0xff)
#define RGB_GREEN(rgb)		(((rgb) >> 8) & 0xff)
#define RGB_BLUE(rgb)		((rgb) & 0xff)

/* common colors */
#define RGB_BLACK			(MAKE_ARGB(255,0,0,0))
#define RGB_WHITE			(MAKE_ARGB(255,255,255,255))

/***************************************************************************
    INLINE FUNCTIONS
***************************************************************************/

/*-------------------------------------------------
    rgb_to_rgb15 - convert an RGB triplet to
    a 15-bit OSD-specified RGB value
-------------------------------------------------*/

INLINE rgb15_t rgb_to_rgb15(rgb_t rgb)
{
	return ((RGB_RED(rgb) >> 3) << 10) | ((RGB_GREEN(rgb) >> 3) << 5) | ((RGB_BLUE(rgb) >> 3) << 0);
}


/*-------------------------------------------------
    rgb_clamp - clamp an RGB component to 0-255
-------------------------------------------------*/

INLINE Bit8u rgb_clamp(Bit32s value)
{
	if (value < 0)
		return 0;
	if (value > 255)
		return 255;
	return value;
}


/*-------------------------------------------------
    pal1bit - convert a 1-bit value to 8 bits
-------------------------------------------------*/

INLINE Bit8u pal1bit(Bit8u bits)
{
	return (bits & 1) ? 0xff : 0x00;
}


/*-------------------------------------------------
    pal2bit - convert a 2-bit value to 8 bits
-------------------------------------------------*/

INLINE Bit8u pal2bit(Bit8u bits)
{
	bits &= 3;
	return (bits << 6) | (bits << 4) | (bits << 2) | bits;
}


/*-------------------------------------------------
    pal3bit - convert a 3-bit value to 8 bits
-------------------------------------------------*/

INLINE Bit8u pal3bit(Bit8u bits)
{
	bits &= 7;
	return (bits << 5) | (bits << 2) | (bits >> 1);
}


/*-------------------------------------------------
    pal4bit - convert a 4-bit value to 8 bits
-------------------------------------------------*/

INLINE Bit8u pal4bit(Bit8u bits)
{
	bits &= 0xf;
	return (bits << 4) | bits;
}


/*-------------------------------------------------
    pal5bit - convert a 5-bit value to 8 bits
-------------------------------------------------*/

INLINE Bit8u pal5bit(Bit8u bits)
{
	bits &= 0x1f;
	return (bits << 3) | (bits >> 2);
}


/*-------------------------------------------------
    pal6bit - convert a 6-bit value to 8 bits
-------------------------------------------------*/

INLINE Bit8u pal6bit(Bit8u bits)
{
	bits &= 0x3f;
	return (bits << 2) | (bits >> 4);
}


/*-------------------------------------------------
    pal7bit - convert a 7-bit value to 8 bits
-------------------------------------------------*/

INLINE Bit8u pal7bit(Bit8u bits)
{
	bits &= 0x7f;
	return (bits << 1) | (bits >> 6);
}


#define WORK_MAX_THREADS			16

/* rectangles describe a bitmap portion */
typedef struct _rectangle rectangle;
struct _rectangle
{
	int				min_x;			/* minimum X, or left coordinate */
	int				max_x;			/* maximum X, or right coordinate (inclusive) */
	int				min_y;			/* minimum Y, or top coordinate */
	int				max_y;			/* maximum Y, or bottom coordinate (inclusive) */
};

/* Standard MIN/MAX macros */
#ifndef MIN
#define MIN(x,y)			((x) < (y) ? (x) : (y))
#endif
#ifndef MAX
#define MAX(x,y)			((x) > (y) ? (x) : (y))
#endif


inline float u2f(Bitu v)
{
	union {
		float ff;
		Bitu vv;
	} u;
	u.vv = v;
	return u.ff;
}


/* Macros for normalizing data into big or little endian formats */
#define FLIPENDIAN_INT16(x)	(((((Bit16u) (x)) >> 8) | ((x) << 8)) & 0xffff)
#define FLIPENDIAN_INT32(x)	((((Bit32u) (x)) << 24) | (((Bit32u) (x)) >> 24) | \
	(( ((Bit32u) (x)) & 0x0000ff00) << 8) | (( ((Bit32u) (x)) & 0x00ff0000) >> 8))
#define FLIPENDIAN_INT64(x)	\
	(												\
		(((((Bit64u) (x)) >> 56) & ((Bit64u) 0xFF)) <<  0)	|	\
		(((((Bit64u) (x)) >> 48) & ((Bit64u) 0xFF)) <<  8)	|	\
		(((((Bit64u) (x)) >> 40) & ((Bit64u) 0xFF)) << 16)	|	\
		(((((Bit64u) (x)) >> 32) & ((Bit64u) 0xFF)) << 24)	|	\
		(((((Bit64u) (x)) >> 24) & ((Bit64u) 0xFF)) << 32)	|	\
		(((((Bit64u) (x)) >> 16) & ((Bit64u) 0xFF)) << 40)	|	\
		(((((Bit64u) (x)) >>  8) & ((Bit64u) 0xFF)) << 48)	|	\
		(((((Bit64u) (x)) >>  0) & ((Bit64u) 0xFF)) << 56)		\
	)

#define ACCESSING_BITS_0_15				((mem_mask & 0x0000ffff) != 0)
#define ACCESSING_BITS_16_31			((mem_mask & 0xffff0000) != 0)

// constants for expression endianness
enum endianness_t
{
	ENDIANNESS_LITTLE,
	ENDIANNESS_BIG
};
const endianness_t ENDIANNESS_NATIVE = ENDIANNESS_LITTLE;
#define ENDIAN_VALUE_LE_BE(endian,leval,beval)	(((endian) == ENDIANNESS_LITTLE) ? (leval) : (beval))
#define NATIVE_ENDIAN_VALUE_LE_BE(leval,beval)	ENDIAN_VALUE_LE_BE(ENDIANNESS_NATIVE, leval, beval)
#define BYTE4_XOR_LE(a) 				((a) ^ NATIVE_ENDIAN_VALUE_LE_BE(0,3))

#define BYTE_XOR_LE(a)  				((a) ^ NATIVE_ENDIAN_VALUE_LE_BE(0,1))

#define profiler_mark_start(x)	do { } while (0)
#define profiler_mark_end()		do { } while (0)


/* Highly useful macro for compile-time knowledge of an array size */
#define ARRAY_LENGTH(x)		(sizeof(x) / sizeof(x[0]))

#define mul_32x32_shift _mul_32x32_shift
INLINE Bit32s _mul_32x32_shift(Bit32s a, Bit32s b, Bit8s shift)
{
	Bit32s result;

#if defined _MSC_VER
    __asm
    {
        mov   eax,a
        imul  b
        mov   cl,shift
        shrd  eax,edx,cl
        mov   result,eax
    }
#else
	Bit64s tmp = (Bit64s)a * (Bit64s)b;
	tmp >>= shift;
	result = (Bit32s)tmp;
#endif

	return result;
}


typedef void (*poly_draw_scanline_func)(void *dest, Bit32s scanline, const poly_extent *extent, const void *extradata, int threadid);

INLINE rgb_t rgba_bilinear_filter(rgb_t rgb00, rgb_t rgb01, rgb_t rgb10, rgb_t rgb11, Bit8u u, Bit8u v)
{
	Bit32u ag0, ag1, rb0, rb1;

	rb0 = (rgb00 & 0x00ff00ff) + ((((rgb01 & 0x00ff00ff) - (rgb00 & 0x00ff00ff)) * u) >> 8);
	rb1 = (rgb10 & 0x00ff00ff) + ((((rgb11 & 0x00ff00ff) - (rgb10 & 0x00ff00ff)) * u) >> 8);
	rgb00 >>= 8;
	rgb01 >>= 8;
	rgb10 >>= 8;
	rgb11 >>= 8;
	ag0 = (rgb00 & 0x00ff00ff) + ((((rgb01 & 0x00ff00ff) - (rgb00 & 0x00ff00ff)) * u) >> 8);
	ag1 = (rgb10 & 0x00ff00ff) + ((((rgb11 & 0x00ff00ff) - (rgb10 & 0x00ff00ff)) * u) >> 8);

	rb0 = (rb0 & 0x00ff00ff) + ((((rb1 & 0x00ff00ff) - (rb0 & 0x00ff00ff)) * v) >> 8);
	ag0 = (ag0 & 0x00ff00ff) + ((((ag1 & 0x00ff00ff) - (ag0 & 0x00ff00ff)) * v) >> 8);

	return ((ag0 << 8) & 0xff00ff00) | (rb0 & 0x00ff00ff);
}

typedef struct _poly_vertex poly_vertex;
struct _poly_vertex
{
	float		x;							/* X coordinate */
	float		y;							/* Y coordinate */
	float		p[MAX_VERTEX_PARAMS];		/* interpolated parameter values */
};


typedef struct _tri_extent tri_extent;
struct _tri_extent
{
	Bit16s		startx;						/* starting X coordinate (inclusive) */
	Bit16s		stopx;						/* ending X coordinate (exclusive) */
};

/* tri_work_unit is a triangle-specific work-unit */
typedef struct _tri_work_unit tri_work_unit;
struct _tri_work_unit
{
//	work_unit_shared	shared;					/* shared data */
	tri_extent			extent[8]; /* array of scanline extents */
};

#endif