#include "save_state.h"
#include "zlib.h"
#ifdef WIN32
#include "direct.h"
#endif
#include "cross.h"
#include <malloc.h>
#include <cstring>
#include <fstream>

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string>
#include <stdlib.h>

SaveState& SaveState::instance() {
    static SaveState singleton;
    return singleton;
}

void SaveState::registerComponent(const std::string& uniqueName, Component& comp) {
    components.insert(std::make_pair(uniqueName, CompData(comp)));
}

namespace Util {
std::string compress(const std::string& input) { //throw (SaveState::Error)
    if (input.empty())
        return input;

    const uLong bufferSize = ::compressBound(input.size());

    std::string output;
    output.resize(bufferSize);

    uLongf actualSize = bufferSize;
    if (::compress2(reinterpret_cast<Bytef*>(&output[0]), &actualSize,
                    reinterpret_cast<const Bytef*>(input.c_str()), input.size(), Z_BEST_SPEED) != Z_OK)
        throw SaveState::Error("Compression failed!");

    output.resize(actualSize);

    //save size of uncompressed data
    const size_t uncompressedSize = input.size(); //save size of uncompressed data
    output.resize(output.size() + sizeof(uncompressedSize)); //won't trigger a reallocation
    ::memcpy(&output[0] + output.size() - sizeof(uncompressedSize), &uncompressedSize, sizeof(uncompressedSize));

    return std::string(&output[0], output.size()); //strip reserved space
}

std::string decompress(const std::string& input) { //throw (SaveState::Error)
    if (input.empty())
        return input;

    //retrieve size of uncompressed data
    size_t uncompressedSize = 0;
    ::memcpy(&uncompressedSize, &input[0] + input.size() - sizeof(uncompressedSize), sizeof(uncompressedSize));

    std::string output;
    output.resize(uncompressedSize);

    uLongf actualSize = uncompressedSize;
    if (::uncompress(reinterpret_cast<Bytef*>(&output[0]), &actualSize,
                     reinterpret_cast<const Bytef*>(input.c_str()), input.size() - sizeof(uncompressedSize)) != Z_OK)
        throw SaveState::Error("Decompression failed!");

    output.resize(actualSize); //should be superfluous!

    return output;
}
}

inline void SaveState::RawBytes::set(const std::string& stream) {
    bytes = stream;
    isCompressed = false;
    dataExists   = true;
}

inline std::string SaveState::RawBytes::get() const { //throw (Error){
    if (isCompressed)
	(Util::decompress(bytes)).swap(bytes);
    isCompressed = false;
    return bytes;
}

inline void SaveState::RawBytes::compress() const { //throw (Error)
    if (!isCompressed)
	(Util::compress(bytes)).swap(bytes);
    isCompressed = true;
}

inline bool SaveState::RawBytes::dataAvailable() const {
    return dataExists;
}

void SaveState::save(size_t slot) { //throw (Error)
    if (slot >= SLOT_COUNT)  return;
    extern unsigned int MEM_TotalPages(void);
    if((MEM_TotalPages()*4096/1024/1024)>200) throw Error("200 MB is the maxium memory size for saving/loading states.");
    bool create_title=false;
    bool create_memorysize=false;
    extern const char* RunningProgram;
	std::string path;
    struct stat status;
	bool Get_Custom_SaveDir(std::string& savedir);
	if(Get_Custom_SaveDir(path)) {
		path+="\\";
	} else {
    // check if dosbox.conf exists in current directory
    stat( "dosbox.conf", &status );
    if ( status.st_mode & S_IFREG ) {
          // path exists in current directory
          path="./save/";
    } else {
          // makes save to appdata
          Cross::CreatePlatformConfigDir(path);
          Cross::GetPlatformConfigDir(path);
          path += "save/";
		  // check if save directory exists in appdata
          stat( path.c_str(), &status );
          if ( status.st_mode & S_IFDIR ) {
                // exists
		  } else {
				Cross::CreateDir(path);
		  }
    }
	}
    for(int i = 1; i <= 10; i++) {
         std::string s;
         std::string current_path;
         std::stringstream out;
  	     out << i;
		 s = out.str();
         current_path = path + s;
		 Cross::CreateDir(current_path);
    }

    try {
        for (CompEntry::iterator i = components.begin(); i != components.end(); ++i) {
            std::ostringstream ss;
            i->second.comp.getBytes(ss);
            i->second.rawBytes[slot].set(ss.str());
            std::string temp = path;
			std::string s;
            std::stringstream out;
            int realslot = slot+1;
			out << realslot;
			s = out.str();
			temp += s;
			temp += "/";

            if(!create_title) {
                  char program_temp[32767];
                  strcpy(program_temp,temp.c_str());
                  strcat(program_temp,"Program Name");
                  std::ofstream programname (program_temp, std::ofstream::binary);
                  programname << RunningProgram;
                  create_title=true;
            }

			if(!create_memorysize) {
                  char memorysize_temp[32767];
                  strcpy(memorysize_temp,temp.c_str());
                  strcat(memorysize_temp,"Memory Size");
				  std::ofstream memorysize (memorysize_temp, std::ofstream::binary);
				  memorysize << MEM_TotalPages();
				  create_memorysize=true;
			}

            char realtemp[32767];
			strcpy(realtemp,temp.c_str());
            strcat(realtemp,i->first.c_str());
            std::ofstream outfile (realtemp, std::ofstream::binary);
            outfile << (Util::compress(ss.str()));
            //outfile << ss.str();
            //compress all other saved states except position "slot"
            //const std::vector<RawBytes>& rb = i->second.rawBytes;
            //std::for_each(rb.begin(), rb.begin() + slot, std::mem_fun_ref(&RawBytes::compress));
            //std::for_each(rb.begin() + slot + 1, rb.end(), std::mem_fun_ref(&RawBytes::compress));
            outfile.close();
            if(outfile.fail()) throw Error("Saved failed!");
        }
    }
    catch (const std::bad_alloc&) {
        throw Error("Save failed! Out of Memory!");
    }
}

void SaveState::load(size_t slot) const { //throw (Error)
//    if (isEmpty(slot)) return;
    extern unsigned int MEM_TotalPages(void);
    if((MEM_TotalPages()*4096/1024/1024)>200) throw Error("200 MB is the maxium memory size for saving/loading states.");
    extern const char* RunningProgram;
    bool read_title=false;
    bool read_memorysize=false;
	std::string path;
    struct stat status;
	bool Get_Custom_SaveDir(std::string& savedir);
	if(Get_Custom_SaveDir(path)) {
		path+="\\";
	} else {
    // check if dosbox.conf exists in current directory
    stat( "dosbox.conf", &status );
    if ( status.st_mode & S_IFREG ) {
          // path exists in current directory
          path="./save/";
    } else {
          // makes save to appdata
          Cross::GetPlatformConfigDir(path);
          path += "save/";
    }
	}

    for (CompEntry::const_iterator i = components.begin(); i != components.end(); ++i) {
        std::filebuf * fb;
        std::ifstream ss;
        std::ifstream check_file;
        fb = ss.rdbuf();
        std::string temp = path;
		std::string s;
        std::stringstream out;
        int realslot = slot+1;
		out << realslot;
		s = out.str();
		temp += s;
		temp += "/";
		
        if(!read_title) {
            std::string program_temp;
            std::fstream check_title;
            int length = 8;

            program_temp = temp;
            program_temp += "Program Name";
            check_title.open(program_temp.c_str(), std::ifstream::in);
            if(check_title.fail()) throw Error("No saved file for this slot. Aborted.");
            check_title.seekg (0, std::ios::end);
            length = check_title.tellg();
            check_title.seekg (0, std::ios::beg);

            char * const buffer = (char*)alloca( (length+1) * sizeof(char)); // char buffer[length];
            check_title.read (buffer, length);
            check_title.close();
            if(strncmp(buffer,RunningProgram,length)) {
	    buffer[length]='\0';
                throw Error("Aborted. Check your program name: ") + buffer;
            }
            read_title=true;
        }

        if(!read_memorysize) {
            std::string memorysize_temp;
            std::fstream check_memorysize;
            int length = 8;

            memorysize_temp = temp;
            memorysize_temp += "Memory Size";
            check_memorysize.open(memorysize_temp.c_str(), std::ifstream::in);
            if(check_memorysize.fail()) throw Error("No saved file for this slot. Aborted.");
            check_memorysize.seekg (0, std::ios::end);
            length = check_memorysize.tellg();
            check_memorysize.seekg (0, std::ios::beg);

            char * const buffer = (char*)alloca( (length+1) * sizeof(char)); // char buffer[length];
            check_memorysize.read (buffer, length);
            check_memorysize.close();
            char str[10];
            itoa(MEM_TotalPages(), str, 10);
            if(strncmp(buffer,str,length)) {
	            buffer[length]='\0';
                throw Error("Aborted. Check your memory size.");
            }
            read_memorysize=true;
        }

		char realtemp[32767];
		strcpy(realtemp,temp.c_str());
        strcat(realtemp,i->first.c_str());
        check_file.open(realtemp, std::ifstream::in);
        check_file.close();
        if(check_file.fail()) throw Error("No saved file for this slot. Aborted.");

        fb->open(realtemp,std::ios::in | std::ios::binary);
		std::string str((std::istreambuf_iterator<char>(ss)), std::istreambuf_iterator<char>());
		std::stringstream mystream;
		mystream << (Util::decompress(str));
		i->second.comp.setBytes(mystream);
        if (mystream.rdbuf()->in_avail() != 0 || mystream.eof()) { //basic consistency check
            throw Error("Save state corrupted! Program in inconsistent state!") + " - " + i->first;
        }
        //compress all other saved states except position "slot"
        //const std::vector<RawBytes>& rb = i->second.rawBytes;
        //std::for_each(rb.begin(), rb.begin() + slot, std::mem_fun_ref(&RawBytes::compress));
        //std::for_each(rb.begin() + slot + 1, rb.end(), std::mem_fun_ref(&RawBytes::compress));
        fb->close();
    }
}

bool SaveState::isEmpty(size_t slot) const {
    if (slot >= SLOT_COUNT) return true;
    return (components.empty() || !components.begin()->second.rawBytes[slot].dataAvailable());
}
