import numpy as np
import pandas as pd

N = 7  # numero di fuochi/switch

def toggle(state, i):
    """
    Toggle secondo il coupling: switch i (1..7)
    inverte {i, i+3, i+4} (mod 7).
    """
    idx = [(i-1) % N, (i+2) % N, (i+3) % N]  # indici 0-based
    new_state = state.copy()
    for j in idx:
        new_state[j] ^= 1  # XOR = toggle in GF(2)
    return new_state

def run_sequence(seq):
    state = np.zeros(N, dtype=int)  # tutti spenti
    history = [state.copy()]
    for s in seq:
        state = toggle(state, s)
        history.append(state.copy())
    return state, np.array(history)

# Alcune sequenze da testare
sequences = {
    "clockwise 1..7": list(range(1, 8)),
    "counterclockwise 7..1": list(range(7, 0, -1)),
    "random perm": [1, 4, 2, 7, 5, 6, 3],
    "random perm 2": [5, 3, 1, 7, 2, 6, 4],
    "random perm 3": [3, 4, 7, 2, 5, 6, 1],
    "skip 6": [1, 2, 3, 4, 5, 7],
    "double press 3": [1, 2, 3, 4, 5, 6, 7, 3],
}

rows = []
for name, seq in sequences.items():
    final_state, _ = run_sequence(seq)
    rows.append({
        "sequence_name": name,
        "sequence": seq,
        "final_state_bits": "".join(map(str, final_state.tolist())),
        "all_on?": bool(final_state.sum() == N),
    })

df = pd.DataFrame(rows)
print(df)

# Costruzione matrice 7x7 del coupling
first_row = np.array([1,0,0,1,1,0,0], dtype=int)
A = np.array([np.roll(first_row, i) for i in range(N)])
print("\nMatrice A =\n", A)

# Rank su GF(2)
def gf2_rank(A):
    A = A.copy() % 2
    m, n = A.shape
    rank = 0
    row = 0
    for col in range(n):
        pivot = None
        for r in range(row, m):
            if A[r, col] % 2 == 1:
                pivot = r
                break
        if pivot is None:
            continue
        if pivot != row:
            A[[row, pivot]] = A[[pivot, row]]
        for r in range(m):
            if r != row and A[r, col] % 2 == 1:
                A[r] ^= A[row]
        row += 1
        rank += 1
        if row == m:
            break
    return rank

# Risoluzione A x = b (mod 2)
def gf2_solve(A, b):
    A = A.copy() % 2
    b = b.copy() % 2
    m, n = A.shape
    M = np.hstack([A, b.reshape(-1, 1)])
    row = 0
    pivots = []
    for col in range(n):
        pivot = None
        for r in range(row, m):
            if M[r, col] == 1:
                pivot = r
                break
        if pivot is None:
            continue
        if pivot != row:
            M[[row, pivot]] = M[[pivot, row]]
        for r in range(m):
            if r != row and M[r, col] == 1:
                M[r, :] ^= M[row, :]
        pivots.append((row, col))
        row += 1
        if row == m:
            break
    x = np.zeros(n, dtype=int)
    for r, c in pivots:
        x[c] = M[r, -1]
    return x

rankA = gf2_rank(A)
print("\nRank(A) su GF(2) =", rankA)

b = np.ones(N, dtype=int)
x = gf2_solve(A, b)
print("Soluzione di A x = 1 (mod 2):", x)
print("Check (A x) mod 2 =", (A @ x) % 2)
