-- FILE: Levels/Common.lua

local Common = {}

--------------------------------------------------------------------------------
-- WEAPON HUD
--------------------------------------------------------------------------------

local WEAPON_UI = {
    [Objects.WeaponType.PISTOLS]  = { label = "Pistols",  y = 50,  color = Color(255, 255,   0, 255) },
    [Objects.WeaponType.UZI]      = { label = "Uzi",      y = 70,  color = Color(  0, 200, 255, 255) },
    [Objects.WeaponType.SHOTGUN]  = { label = "Shotgun",  y = 70,  color = Color(255, 100, 100, 255) },
    [Objects.WeaponType.HK]       = { label = "HK",       y = 70,  color = Color(255, 180,  60, 255) },
}

function Common.DrawWeaponHUD()
    local wt = Objects.LaraObject.GetWeaponType(Lara)
    if wt ~= Objects.WeaponType.PISTOLS then
        local status = Lara:GetHandStatus()
        if status == 4 then -- WEAPON_READY
            local ui    = WEAPON_UI[wt]
            local label = (ui and ui.label) or tostring(wt)
            local y     = (ui and ui.y)     or 60
            local col   = (ui and ui.color) or Color(255,255,255,255)

            if wt == Objects.WeaponType.SHOTGUN then
                local at = Lara:GetAmmoType()
                local isWide = (at == Objects.AmmoType.SHOTGUN_WIDE)
                label = string.format("%s (%s)", label, isWide and "W" or "N")
                col = isWide and Color(0,160,255,255) or Color(255,80,80,255)
            end

            local ammo  = Lara:GetAmmoCount()
            local text  = string.format("%s: %s", label, tostring(ammo))
            local s = DisplayString(text, Vec2(40, y), 1.0, col, false)
            Strings.ShowString(s, 0.1, true)
        end
    end
end
--------------------------------------------------------------------------------


--------------------------------------------------------------------------------
-- CUSTOM FLARE
--------------------------------------------------------------------------------

Common.flare_on = false
Common.flare_pitch = 0

function Common.UpdateFlare(dt)
    if Input.IsKeyHit(Input.ActionID.FLARE) then
        Common.flare_on = not Common.flare_on
        Sound.PlaySound(379)
    end

    if not Common.flare_on then return end

    if Input.IsKeyHit(Input.ActionID.L) then Common.flare_pitch = Common.flare_pitch - 0.1 end
    if Input.IsKeyHit(Input.ActionID.K) then Common.flare_pitch = 0 end
    if Input.IsKeyHit(Input.ActionID.J) then Common.flare_pitch = Common.flare_pitch + 0.1 end

    local pos = Lara:GetJointPosition(11, Vec3(-32, -64, 64))
    local rot = Lara:GetRotation()
    local dir = rot:Direction()
    dir = dir + Vec3(0, Common.flare_pitch, 0)
    dir:Normalize()
    pos = pos + dir * 16

    Effects.EmitSpotLight(pos, dir, Color(255, 250, 220), 12, 6, 24, false, "test_spot")
    Effects.EmitLight(pos, Color(200,180,160), 8, 12, "flare_fill")
end

--------------------------------------------------------------------------------


--------------------------------------------------------------------------------
-- STATISTICHE DI FINE LIVELLO
--------------------------------------------------------------------------------

Common.level_time = 0
Common.stats_active = false

function Common.UpdateLevelTime(dt)
    Common.level_time = Common.level_time + dt
end

function Common.ShowStatsHUD()
    Common.stats_active = true
    Flow.SetFreezeMode(Flow.FreezeMode.FULL)
    Input.ClearAllKeys()

    Sound.PlayAudioTrack("datr3finelivello", Sound.SoundTrackType.ONESHOT)

    local lvl = Flow.GetCurrentLevel()
    local levelName = Flow.GetString(lvl.nameKey)
    local stats = Flow.GetStatistics()

    local lt = Common.level_time or 0
    local h = math.floor(lt / 3600)
    local m = math.floor((lt % 3600) / 60)
    local s = math.floor(lt % 60)
    local time_str = string.format("%02d:%02d:%02d", h, m, s)

    local secrets_level_total  = tonumber(lvl.secrets or 0)
    local secrets_level_found  = tonumber(stats.secrets or 0)
    local secrets_game_found   = tonumber(Flow.GetSecretCount() or 0)
    local secrets_game_total   = tonumber(Flow.GetTotalSecretCount() or 0)
    local kills    = tonumber(stats.kills or 0)
    local ammoUsed = tonumber(stats.ammoUsed or 0)

    local function show(str, y, color)
        local s = DisplayString(str, Vec2(PercentToScreen(50, y)), 1.0, color)
        s:SetFlags{TEN.Strings.DisplayStringOption.CENTER, TEN.Strings.DisplayStringOption.SHADOW}
        Strings.ShowString(s, 300, true)
    end

    show(levelName, 47, Color(165,42,42))
    show("Statistics", 55, Color(150,200,255))
    show("========================", 63, Color(150,200,255))

    local text = string.format(
        "Time: %s\nSecrets: %d/%d (%d/%d)\nKills: %d\nAmmo used: %d",
        time_str, secrets_level_found, secrets_level_total,
        secrets_game_found, secrets_game_total,
        kills, ammoUsed
    )
    show(text, 70, Color(200,220,255))
end

function Common.OnFreeze()
    if not Common.stats_active then return end
    if Flow.GetFreezeMode() ~= Flow.FreezeMode.FULL then return end
    if Input.IsKeyHeld(Input.ActionID.RETURN) or Input.IsKeyHeld(Input.ActionID.ESCAPE) then
        Common.stats_active = false
        Flow.SetFreezeMode(Flow.FreezeMode.NONE)
        Flow.EndLevel()
    end
end

--------------------------------------------------------------------------------

return Common
