-- FILE: \Engine\CutsceneScript.lua
local Timer = require("Engine.Timer")

local CutsceneScript

LevelFuncs.Engine.CutsceneScript = {}
LevelVars.Engine.CutsceneScript = {cutscene_scripts = {}}

CutsceneScript = {
	Create = function(name, loop, ...)
		print("Creating cutscene script")
		local obj = {}
		local mt = {}
		mt.__index = CutsceneScript
		setmetatable(obj, mt)
		
		obj.name = name
		LevelVars.Engine.CutsceneScript.cutscene_scripts[name] = {} 
		local thisCS = LevelVars.Engine.CutsceneScript.cutscene_scripts[name]
		thisCS.name = name
		thisCS.timesFuncsAndArgs = {...}
		thisCS.loop = loop
		
		local tfa = thisCS.timesFuncsAndArgs
		thisCS.timers = {}
		
		for i = 1, #tfa, 2 do
			local timerIndex = #thisCS.timers + 1 
			local timerName = "__TEN_cutsceneScript_" .. name .. "_timer" .. timerIndex 
			print("Getting func + args")
			local funcAndArgs = tfa[i+1]
			local func
			
			if type(funcAndArgs) == "userdata" then
				-- we only have a function
				func = funcAndArgs
				funcAndArgs = {}
			else
				-- we have a function and possible args
				func = table.remove(funcAndArgs, 1)
			end
			
			local thisTimer = Timer.Create(timerName,
						tfa[i], -- time
						false,
						false,
						func,
						table.unpack(funcAndArgs) -- now with func removed
						)

			thisCS.timers[timerIndex] = timerName
			
		end
		return obj
	end;
		
	--- Get an cutscene script by its name.
	-- @string name The label that was given to the script when it was created
	-- @treturn CutsceneScript The script
	Get = function(name)
		if LevelVars.Engine.CutsceneScript.cutscene_scripts[name] then
			local obj = {}
			local mt = {}
			mt.__index = CutsceneScript
			setmetatable(obj, mt)
			obj.name = name
			return obj
		end
		return nil
	end;
	
	-- @function myscript:SetPaused
	-- @bool p if true, the script will be paused; if false, it will be unpaused 
	SetPaused = function(t, p)
		local thisCS = LevelVars.Engine.CutsceneScript.cutscene_scripts[t.name]

		for i = 1, #thisCS.timers do
			Timer.Get(thisCS.timers[i]):SetPaused(p)
		end
	end;
	
	
	--- Get whether or not the script is paused
	-- @function myscript:IsPaused
	-- @treturn bool true if the timer is paused, false if otherwise
	IsPaused = function(t)
		local thisCS = LevelVars.Engine.CutsceneScript.cutscene_scripts[t.name]
		return Timer.Get(thisCS.timers[0]):IsPaused()
	end;
	
	--- Begin or unpause a script. If showing the remaining time on-screen, its color will be set to white.
	-- @function myscript:Start
	Start = function(t)
		local thisCS = LevelVars.Engine.CutsceneScript.cutscene_scripts[t.name]
		
		for i = 1, #thisCS.timers do
			Timer.Get(thisCS.timers[i]):Start()
		end
	end;

	--- Stop the script.
	--@function myscript:Stop
	Stop = function(t)
		local thisCS = LevelVars.Engine.CutsceneScript.cutscene_scripts[t.name]
		
		for i = 1, #thisCS.timers do
			Timer.Get(thisCS.timers[i]):Stop()
		end
	end;
	
	StopEverythingElse = function(t)
		for i,cutscene_script in pairs(LevelVars.Engine.CutsceneScript.cutscene_scripts) do
			if i ~= t.name then
				--local cs = LevelVars.Engine.CutsceneScript.Get(i)
				print("Stopping " .. i .. " for " .. t.name .. cutscene_script)
				local thisCS = LevelVars.Engine.CutsceneScript.cutscene_scripts[name]
				for i = 1, #thisCS.timers do
					Timer.Get(thisCS.timers[i]):Stop()
				end
			end
		end
	end;
}

return CutsceneScript