local socket = require("socket")

local Config = {
    HOST = "basic-multiplayer.fly.dev",
    PORT = 10001,
    SOCKET_TIMEOUT = 2,
    ID_RECEIVE_TIMEOUT = 3,
    FORCE_SYNC_INTERVAL = 2,
    RECONNECT_INTERVAL = 60,
    CONNECTION_STATUS_LOG_INTERVAL = 300,
    DEBUG_LOG_INTERVAL = 60,
    DEBUG_LOG_INTERVAL_LONG = 180,
    POS_CHANGE_THRESHOLD = 1.0,
    ROT_CHANGE_THRESHOLD = 0.05,
    LOOK_ROTATION_THRESHOLD = 0.5,
    MAX_MESSAGES_PER_FRAME = 50
}

local ConnectionState = {
    client = nil,
    isConnected = false,
    myId = nil,
    isSaving = false,
    serverMaxPlayers = 10
}

local SyncState = {
    remotePlayers = {},
    remoteVehicles = {},
    lastSentState = nil,
    frameCounter = 0
}

local SoundState = {
    pendingSoundID = 0,
    lastSentSoundID = 0,
    lastReceivedSoundID = {},
    lastSoundFrame = {}
}

local function splitString(inputstr, sep)
    if sep == nil then sep = "%s" end
    local t = {}
    for str in string.gmatch(inputstr, "([^" .. sep .. "]+)") do
        table.insert(t, str)
    end
    return t
end

local function trimMessage(msg)
    return msg:gsub("[\r\n]+$", "")
end

local function connect_impl()
    if ConnectionState.isConnected then return end

    local s, serr = socket.tcp()
    if not s then
        print("[ERROR] Failed to create TCP socket: " .. tostring(serr))
        return
    end

    s:settimeout(Config.SOCKET_TIMEOUT)
    local ok, cerr = s:connect(Config.HOST, Config.PORT)

    if ok then
        ConnectionState.client = s

        ConnectionState.client:settimeout(Config.ID_RECEIVE_TIMEOUT)
        local idMsg, idErr = ConnectionState.client:receive("*l")
        if idMsg then
            idMsg = trimMessage(idMsg)

            if idMsg == "FULL" then
                print("[ERROR] Server is full")
                pcall(function() s:close() end)
                ConnectionState.client = nil
                return
            end

            if idMsg:match("^ID:") then
                local id, _, maxPlayers = idMsg:match("ID:(%d+):SLOT:(%d+):MAX:(%d+)")
                if id then
                    ConnectionState.myId = tonumber(id)
                    ConnectionState.serverMaxPlayers = tonumber(maxPlayers) or 10
                    ConnectionState.isConnected = true
                end
            end
        else
            print("[ERROR] Timeout waiting for ID: " .. tostring(idErr))
            pcall(function() s:close() end)
            ConnectionState.client = nil
            return
        end
    else
        print("[ERROR] Failed to connect: " .. tostring(cerr))
        pcall(function() s:close() end)
        ConnectionState.client = nil
    end
end

local function getLaraState()
    local lara = Lara
    if not lara then return nil end

    local objID = lara:GetObjectID()
    if objID ~= 0 then
        return nil
    end

    local ok, stateData = pcall(function() return lara:GetFullState() end)
    if not ok or not stateData then
        print("[ERROR] GetFullState failed")
        return nil
    end

    local pos = lara:GetPosition()
    local rot = lara:GetRotation()
    stateData.posX = pos.x
    stateData.posY = pos.y
    stateData.posZ = pos.z
    stateData.rotY = rot.y
    stateData.rotX = rot.x or 0
    stateData.rotZ = rot.z or 0

    stateData.health = lara:GetHP()
    stateData.ammo = lara:GetAmmoCount()

    stateData.inputForward = KeyIsHeld(ActionID.FORWARD)
    stateData.inputBack = KeyIsHeld(ActionID.BACK)
    stateData.inputLeft = KeyIsHeld(ActionID.LEFT)
    stateData.inputRight = KeyIsHeld(ActionID.RIGHT)
    stateData.inputJump = KeyIsHeld(ActionID.JUMP)
    stateData.inputAction = KeyIsHeld(ActionID.ACTION)
    stateData.inputCrouch = KeyIsHeld(ActionID.CROUCH)
    stateData.inputWalk = KeyIsHeld(ActionID.WALK)
    stateData.inputSprint = KeyIsHeld(ActionID.SPRINT)
    stateData.inputRoll = KeyIsHeld(ActionID.ROLL)
    stateData.inputLook = KeyIsHeld(ActionID.LOOK)
    stateData.inputFlare = KeyIsHeld(ActionID.FLARE)
    stateData.inputStepLeft = false
    stateData.inputStepRight = false
    stateData.inputDraw = KeyIsHeld(ActionID.DRAW)

    return stateData
end

local function getLocalVehicleState()
    local vehicle = Lara:GetVehicle()
    if not vehicle then return nil end

    local rot = vehicle:GetRotation()
    return {
        objectID = vehicle:GetObjectID(),
        posX = math.floor(vehicle:GetPosition().x),
        posY = math.floor(vehicle:GetPosition().y),
        posZ = math.floor(vehicle:GetPosition().z),
        rotX = rot.x,
        rotY = rot.y,
        rotZ = rot.z
    }
end

local function serializeVehicle(vehicleState)
    if not vehicleState then return "" end

    return string.format(":VEH:%d:%d:%d:%d:%.2f:%.2f:%.2f",
        vehicleState.objectID,
        vehicleState.posX,
        vehicleState.posY,
        vehicleState.posZ,
        vehicleState.rotX,
        vehicleState.rotY,
        vehicleState.rotZ)
end

local function serializeRotationXZ(state)
    if math.abs(state.rotX or 0) > 0.01 or math.abs(state.rotZ or 0) > 0.01 then
        return string.format(":ROT:%.2f:%.2f", state.rotX, state.rotZ)
    end
    return ""
end

local function serializeSound()
    local soundSuffix = ":SND:" .. SoundState.pendingSoundID
    SoundState.lastSentSoundID = SoundState.pendingSoundID
    SoundState.pendingSoundID = 0
    return soundSuffix
end

local function serializeArmAim()
    local aimSuffix = ""
    pcall(function()
        local leftArmRot = Lara:GetLeftArmOrientation()
        local rightArmRot = Lara:GetRightArmOrientation()
        local targetArmRot = Lara:GetTargetArmOrientation()

        local leftArmLocked = Lara:GetLeftArmLocked() and 1 or 0
        local rightArmLocked = Lara:GetRightArmLocked() and 1 or 0

        aimSuffix = string.format(
            ":AIM:%.2f:%.2f:%.2f:%.2f:%.2f:%.2f:%.2f:%.2f:%.2f:%d:%d",
            leftArmRot.x, leftArmRot.y, leftArmRot.z,
            rightArmRot.x, rightArmRot.y, rightArmRot.z,
            targetArmRot.x, targetArmRot.y, targetArmRot.z,
            leftArmLocked, rightArmLocked
        )
    end)

    return aimSuffix
end

local function serializeLookState()
    local lookSuffix = ":LOOK:0:0.00:0.00"
    pcall(function()
        local headRot = Lara:GetExtraHeadRotation()

        local lookActive = math.abs(headRot.x) > Config.LOOK_ROTATION_THRESHOLD or
                          math.abs(headRot.y) > Config.LOOK_ROTATION_THRESHOLD

        if lookActive then
            local orientX = headRot.x * 2
            local orientY = headRot.y * 2
            lookSuffix = string.format(":LOOK:%d:%.2f:%.2f", 1, orientX, orientY)
        else
            lookSuffix = ":LOOK:0:0.00:0.00"
        end
    end)

    return lookSuffix
end

local function serializeHolster(state)
    if state.leftHolster or state.rightHolster or state.backHolster then
        return string.format(":H:%d:%d:%d",
            tonumber(state.leftHolster) or 0,
            tonumber(state.rightHolster) or 0,
            tonumber(state.backHolster) or 0)
    end
    return ""
end

local function hasStateChanged(state, lastState)
    if not lastState then return true end

    if state.animNumber ~= lastState.animNumber then return true end

    if math.abs(state.posX - lastState.posX) > Config.POS_CHANGE_THRESHOLD or
       math.abs(state.posY - lastState.posY) > Config.POS_CHANGE_THRESHOLD or
       math.abs(state.posZ - lastState.posZ) > Config.POS_CHANGE_THRESHOLD then
        return true
    end

    if math.abs(state.rotY - lastState.rotY) > Config.ROT_CHANGE_THRESHOLD or
       math.abs(state.rotX - (lastState.rotX or 0)) > Config.ROT_CHANGE_THRESHOLD or
       math.abs(state.rotZ - (lastState.rotZ or 0)) > Config.ROT_CHANGE_THRESHOLD then
        return true
    end

    if state.inputForward ~= lastState.inputForward or
       state.inputBack ~= lastState.inputBack or
       state.inputLeft ~= lastState.inputLeft or
       state.inputRight ~= lastState.inputRight or
       state.inputJump ~= lastState.inputJump or
       state.inputAction ~= lastState.inputAction or
       state.inputCrouch ~= lastState.inputCrouch or
       state.inputWalk ~= lastState.inputWalk or
       state.inputSprint ~= lastState.inputSprint or
       state.inputRoll ~= lastState.inputRoll or
       state.inputFlare ~= lastState.inputFlare or
       state.inputDraw ~= lastState.inputDraw then
        return true
    end

    return false
end

local function cacheState(state)
    SyncState.lastSentState = {
        posX = state.posX,
        posY = state.posY,
        posZ = state.posZ,
        rotY = state.rotY,
        rotX = state.rotX,
        rotZ = state.rotZ,
        animNumber = state.animNumber,
        inputForward = state.inputForward,
        inputBack = state.inputBack,
        inputLeft = state.inputLeft,
        inputRight = state.inputRight,
        inputJump = state.inputJump,
        inputAction = state.inputAction,
        inputCrouch = state.inputCrouch,
        inputWalk = state.inputWalk,
        inputSprint = state.inputSprint,
        inputRoll = state.inputRoll,
        inputFlare = state.inputFlare,
        inputDraw = state.inputDraw
    }
end

local function buildHybridMessage(state, suffixes)
    local meshString = ""
    if state.meshIndices then
        meshString = table.concat(state.meshIndices, ",")
    end

    local msg = string.format(
        "INPUT:HYBRID:%d:%d:%d:%.2f:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%d:%s%s\n",
        math.floor(state.posX),
        math.floor(state.posY),
        math.floor(state.posZ),
        state.rotY,
        state.animNumber,
        state.frameNumber,
        tonumber(state.gunType) or 0,
        tonumber(state.handStatus) or 0,
        (state.isLow and 1 or 0),
        (state.inputForward and 1 or 0),
        (state.inputBack and 1 or 0),
        (state.inputLeft and 1 or 0),
        (state.inputRight and 1 or 0),
        (state.inputJump and 1 or 0),
        (state.inputAction and 1 or 0),
        (state.inputCrouch and 1 or 0),
        (state.inputWalk and 1 or 0),
        (state.inputSprint and 1 or 0),
        (state.inputRoll and 1 or 0),
        (state.inputDraw and 1 or 0),
        (state.inputFlare and 1 or 0),
        tonumber(state.leftArmAnim) or 0,
        tonumber(state.leftArmFrame) or 0,
        tonumber(state.leftArmFrameBase) or 0,
        tonumber(state.rightArmAnim) or 0,
        tonumber(state.rightArmFrame) or 0,
        tonumber(state.rightArmFrameBase) or 0,
        tonumber(state.animObjectID) or 0,
        tonumber(state.frameBase) or 0,
        (state.isUsingBinoculars and 1 or 0),
        (state.flareControlLeft and 1 or 0),
        0,
        0,
        tonumber(state.leftArmGunFlash) or 0,
        tonumber(state.rightArmGunFlash) or 0,
        tonumber(state.leftArmDoubleSND) or 0,
        tonumber(state.rightArmDoubleSND) or 0,
        meshString,
        suffixes
    )

    return msg
end

local function sendPlayerData()
    if not (ConnectionState.isConnected and ConnectionState.client) then return end

    local state = getLaraState()
    if not state then return end

    local stateChanged = hasStateChanged(state, SyncState.lastSentState)
    if not (stateChanged or (SyncState.frameCounter % Config.FORCE_SYNC_INTERVAL == 0)) then
        return
    end

    cacheState(state)

    local rotSuffix = serializeRotationXZ(state)
    local vehicleSuffix = serializeVehicle(getLocalVehicleState())
    local soundSuffix = serializeSound()
    local aimSuffix = serializeArmAim()
    local lookSuffix = serializeLookState()
    local holsterSuffix = serializeHolster(state)

    local allSuffixes = rotSuffix .. vehicleSuffix .. soundSuffix .. aimSuffix .. lookSuffix .. holsterSuffix

    local msg = buildHybridMessage(state, allSuffixes)

    local okSend, sendErr = pcall(function() return ConnectionState.client:send(msg) end)
    if not okSend or not sendErr then
        print('[ERROR] Network send failed: ' .. tostring(sendErr))
        ConnectionState.isConnected = false
        pcall(function() if ConnectionState.client then ConnectionState.client:close() end end)
        ConnectionState.client = nil
    end
end

local function deserializeVehicleState(parts, startIndex)
    if not parts[startIndex] or parts[startIndex] ~= "VEH" then
        return nil
    end

    local objectID = tonumber(parts[startIndex + 1])
    local posX = tonumber(parts[startIndex + 2])
    local posY = tonumber(parts[startIndex + 3])
    local posZ = tonumber(parts[startIndex + 4])
    local rotX = tonumber(parts[startIndex + 5])
    local rotY = tonumber(parts[startIndex + 6])
    local rotZ = tonumber(parts[startIndex + 7])

    if not (objectID and posX and posY and posZ and rotX and rotY and rotZ) then
        return nil
    end

    return {
        objectID = objectID,
        posX = posX,
        posY = posY,
        posZ = posZ,
        rotX = rotX,
        rotY = rotY,
        rotZ = rotZ
    }
end

local function updateRemoteVehicle(playerId, vehicleState)
    if not vehicleState then
        if SyncState.remoteVehicles[playerId] then
            pcall(function() SyncState.remoteVehicles[playerId]:Destroy() end)
            SyncState.remoteVehicles[playerId] = nil
        end
        return
    end

    local vehicle = SyncState.remoteVehicles[playerId]
    if not vehicle then
        local vehicleName = "remote_vehicle_" .. playerId
        local okCreate, created = pcall(function()
            return TEN.Objects.Moveable(
                vehicleState.objectID,
                vehicleName,
                Vec3(vehicleState.posX, vehicleState.posY, vehicleState.posZ),
                Rotation(vehicleState.rotX, vehicleState.rotY, vehicleState.rotZ)
            )
        end)

        if okCreate and created then
            SyncState.remoteVehicles[playerId] = created
            vehicle = created
        else
            print(string.format("[ERROR] Failed to create vehicle for player %d", playerId))
            return
        end
    end

    pcall(function()
        vehicle:SetPosition(Vec3(vehicleState.posX, vehicleState.posY, vehicleState.posZ))
        vehicle:SetRotation(Rotation(vehicleState.rotX, vehicleState.rotY, vehicleState.rotZ))
    end)
end

local function parseAimData(parts)
    local aimLeftArmRot = nil
    local aimRightArmRot = nil
    local aimTargetArmRot = nil
    local aimLeftArmLocked = false
    local aimRightArmLocked = false

    for i = 1, #parts - 10 do
        if parts[i] == "AIM" then
            aimLeftArmRot = Rotation(
                tonumber(parts[i + 1]) or 0,
                tonumber(parts[i + 2]) or 0,
                tonumber(parts[i + 3]) or 0
            )
            aimRightArmRot = Rotation(
                tonumber(parts[i + 4]) or 0,
                tonumber(parts[i + 5]) or 0,
                tonumber(parts[i + 6]) or 0
            )
            aimTargetArmRot = Rotation(
                tonumber(parts[i + 7]) or 0,
                tonumber(parts[i + 8]) or 0,
                tonumber(parts[i + 9]) or 0
            )
            aimLeftArmLocked = tonumber(parts[i + 10]) == 1
            aimRightArmLocked = tonumber(parts[i + 11]) == 1
            break
        end
    end

    return {
        leftArmRot = aimLeftArmRot,
        rightArmRot = aimRightArmRot,
        targetArmRot = aimTargetArmRot,
        leftArmLocked = aimLeftArmLocked,
        rightArmLocked = aimRightArmLocked
    }
end

local function parseLookData(parts)
    local lookActive = false
    local lookOrientX = 0
    local lookOrientY = 0

    for i = 1, #parts - 2 do
        if parts[i] == "LOOK" then
            lookActive = tonumber(parts[i + 1]) == 1
            lookOrientX = tonumber(parts[i + 2]) or 0
            lookOrientY = tonumber(parts[i + 3]) or 0
            break
        end
    end

    return {
        active = lookActive,
        orientX = lookOrientX,
        orientY = lookOrientY
    }
end

local function parseRotationXZ(parts)
    local rotX = 0
    local rotZ = 0

    for i = 1, #parts - 2 do
        if parts[i] == "ROT" then
            rotX = tonumber(parts[i + 1]) or 0
            rotZ = tonumber(parts[i + 2]) or 0
            break
        end
    end

    return rotX, rotZ
end

local function parseSoundData(parts)
    for i = 1, #parts - 1 do
        if parts[i] == "SND" then
            local soundID = tonumber(parts[i + 1])
            if soundID then
                return soundID
            end
        end
    end
    return nil
end

local function parseHolsterData(parts)
    for i = 1, #parts - 3 do
        if parts[i] == "H" then
            return {
                leftHolster = tonumber(parts[i + 1]) or 0,
                rightHolster = tonumber(parts[i + 2]) or 0,
                backHolster = tonumber(parts[i + 3]) or 0
            }
        end
    end
    return nil
end

local function processSoundSync(parts, double)
    for i = 1, #parts - 1 do
        if parts[i] == "SND" then
            local soundID = tonumber(parts[i + 1]) or 0
            if soundID > 0 then
                local pos = double:GetPosition()
                TEN.Sound.StopSound(soundID)
                TEN.Sound.PlaySound(soundID, pos)
            end
        end
    end
end

local function handlePlayerLeave(msg)
    local id = msg:match("LEAVE:(%d+)")
    id = tonumber(id)

    if SyncState.remotePlayers[id] then
        pcall(function() SyncState.remotePlayers[id]:Destroy() end)
        SyncState.remotePlayers[id] = nil

        if SyncState.remoteVehicles[id] then
            pcall(function() SyncState.remoteVehicles[id]:Destroy() end)
            SyncState.remoteVehicles[id] = nil
        end
    end
end

local function getOrCreateRemotePlayer(id, parts)
    local double = SyncState.remotePlayers[id]
    if double then return double end

    local doubleName = "lara_double_" .. id
    local newPos = Vec3(tonumber(parts[1]), tonumber(parts[2]), tonumber(parts[3]))
    local rotY = tonumber(parts[4])
    local rotX, rotZ = parseRotationXZ(parts)
    local newRot = Rotation(rotX, rotY, rotZ)

    local okCreate, created = pcall(function()
        return TEN.Objects.Moveable(TEN.Objects.ObjID.LARA_DOUBLE, doubleName, newPos, newRot)
    end)

    if okCreate and created then
        SyncState.remotePlayers[id] = created
        return created
    else
        print("[ERROR] Failed to create remote player " .. id)
        return nil
    end
end

local function buildStateTable(parts)
    return {
        animNumber = tonumber(parts[5]),
        frameNumber = tonumber(parts[6]),
        gunType = tonumber(parts[7]),
        handStatus = tonumber(parts[8]),
        isLow = (tonumber(parts[9]) == 1),
        inputForward = (tonumber(parts[10]) == 1),
        inputBack = (tonumber(parts[11]) == 1),
        inputLeft = (tonumber(parts[12]) == 1),
        inputRight = (tonumber(parts[13]) == 1),
        inputJump = (tonumber(parts[14]) == 1),
        inputAction = (tonumber(parts[15]) == 1),
        inputCrouch = (tonumber(parts[16]) == 1),
        inputWalk = (tonumber(parts[17]) == 1),
        inputSprint = (tonumber(parts[18]) == 1),
        inputRoll = (tonumber(parts[19]) == 1),
        inputDraw = (tonumber(parts[20]) == 1),
        inputFlare = (tonumber(parts[21]) == 1),
        leftArmAnim = tonumber(parts[22]),
        leftArmFrame = tonumber(parts[23]),
        leftArmFrameBase = tonumber(parts[24]),
        rightArmAnim = tonumber(parts[25]),
        rightArmFrame = tonumber(parts[26]),
        rightArmFrameBase = tonumber(parts[27]),
        animObjectID = tonumber(parts[28]),
        frameBase = tonumber(parts[29]),
        isUsingBinoculars = (tonumber(parts[30]) == 1),
        flareControlLeft = (tonumber(parts[31]) == 1),
        leftArmGunFlash = tonumber(parts[34]),
        rightArmGunFlash = tonumber(parts[35]),
        leftArmDoubleSND = tonumber(parts[36]) or 0,
        rightArmDoubleSND = tonumber(parts[37]) or 0,
        meshIndices = {}
    }
end

local function parseMeshIndices(parts, stateTable)
    if parts[38] and #parts[38] > 0 then
        local meshParts = splitString(parts[38], ",")
        for i=1, #meshParts do
            stateTable.meshIndices[i] = tonumber(meshParts[i])
        end
    end
end

local function processMessage(msg)
    if msg:match("^LEAVE:") then
        handlePlayerLeave(msg)
        return
    end

    local id, _, payload = msg:match("^(%d+):(%d+):INPUT:(HYBRID:.*)$")
    if not payload then
        return
    end

    id = tonumber(id)
    if ConnectionState.myId and id == ConnectionState.myId then return end

    payload = payload:sub(8)
    local parts = splitString(payload, ":")

    local double = getOrCreateRemotePlayer(id, parts)
    if not double then return end

    local stateTable = buildStateTable(parts)

    local newPos = Vec3(tonumber(parts[1]), tonumber(parts[2]), tonumber(parts[3]))
    local rotY = tonumber(parts[4])
    local rotX, rotZ = parseRotationXZ(parts)
    local newRot = Rotation(rotX, rotY, rotZ)

    pcall(function() double:SetPosition(newPos) end)
    pcall(function() double:SetRotation(newRot) end)

    parseMeshIndices(parts, stateTable)

    local aimData = parseAimData(parts)
    if aimData.leftArmRot then
        stateTable.leftArmOrientation = aimData.leftArmRot
        stateTable.rightArmOrientation = aimData.rightArmRot
        stateTable.targetArmOrientation = aimData.targetArmRot
        stateTable.leftArmLocked = aimData.leftArmLocked
        stateTable.rightArmLocked = aimData.rightArmLocked
    end

    local lookData = parseLookData(parts)
    stateTable.lookActive = lookData.active
    stateTable.lookOrientX = lookData.orientX
    stateTable.lookOrientY = lookData.orientY

    local soundID = parseSoundData(parts)
    if soundID then
        stateTable.soundID = soundID
    end

    local holsterData = parseHolsterData(parts)
    if holsterData then
        stateTable.leftHolster = holsterData.leftHolster
        stateTable.rightHolster = holsterData.rightHolster
        stateTable.backHolster = holsterData.backHolster
    end

    local setStateSuccess, setStateError = pcall(function()
        double:SetLaraDoubleState(stateTable)
    end)

    if not setStateSuccess then
        print("[ERROR] SetLaraDoubleState failed for ID=" .. id .. ": " .. tostring(setStateError))
    end

    local vehicleState = nil
    for i = 1, #parts - 7 do
        if parts[i] == "VEH" then
            vehicleState = deserializeVehicleState(parts, i)
            break
        end
    end
    updateRemoteVehicle(id, vehicleState)

    processSoundSync(parts, double)
end

local function receive()
    if not (ConnectionState.isConnected and ConnectionState.client) then return end

    ConnectionState.client:settimeout(0)
    local msgCount = 0

    while msgCount < Config.MAX_MESSAGES_PER_FRAME do
        local data, err = ConnectionState.client:receive("*l")
        if data then
            msgCount = msgCount + 1
            data = trimMessage(data)
            if #data > 0 then
                local ok, processErr = pcall(processMessage, data)
                if not ok then
                    print("[New_Level] Error processing message: " .. tostring(processErr))
                end
            end
        elseif err == "timeout" or err == "wantread" then
            break
        elseif err then
            print("[ERROR] Network receive error: ".. tostring(err))
            ConnectionState.isConnected = false
            pcall(function() if ConnectionState.client then ConnectionState.client:close() end end)
            ConnectionState.client = nil
            break
        else
            print("[ERROR] Server closed connection")
            ConnectionState.isConnected = false
            pcall(function() if ConnectionState.client then ConnectionState.client:close() end end)
            ConnectionState.client = nil
            break
        end
    end
end

LevelFuncs = LevelFuncs or {}
LevelFuncs.Engine = LevelFuncs.Engine or {}

LevelFuncs.Engine.connect = connect_impl
_G.connect = connect_impl

LevelFuncs.OnLoad = function()
    ConnectionState.myId = nil
    ConnectionState.isConnected = false
    ConnectionState.isSaving = false
    if ConnectionState.client then
        pcall(function() ConnectionState.client:close() end)
        ConnectionState.client = nil
    end

    SyncState.remotePlayers = {}
    SyncState.remoteVehicles = {}
    SyncState.frameCounter = 0
    SyncState.lastSentState = nil

    SoundState.pendingSoundID = 0
    SoundState.lastSentSoundID = 0
    SoundState.lastReceivedSoundID = {}
    SoundState.lastSoundFrame = {}

    LevelFuncs.Engine.connect()
end

LevelFuncs.OnSave = function()
    ConnectionState.isSaving = true

    for id, player in pairs(SyncState.remotePlayers) do
        pcall(function() player:Destroy() end)
    end
    SyncState.remotePlayers = {}

    for id, vehicle in pairs(SyncState.remoteVehicles) do
        pcall(function() vehicle:Destroy() end)
    end
    SyncState.remoteVehicles = {}
end

LevelFuncs.OnStart = function()
    LevelFuncs.Engine.connect()
end

LevelFuncs.OnLoop = function(dt)
    SyncState.frameCounter = SyncState.frameCounter + 1

    if ConnectionState.isSaving then
        ConnectionState.isSaving = false
        return
    end

    if not ConnectionState.isConnected then
        if SyncState.frameCounter % Config.RECONNECT_INTERVAL == 0 then
            LevelFuncs.Engine.connect()
        end
        return
    end

    receive()
    sendPlayerData()
end

LevelFuncs.OnEnd = function()
    if ConnectionState.client then
        pcall(function() ConnectionState.client:close() end)
    end
    ConnectionState.isConnected = false

    for id, player in pairs(SyncState.remotePlayers) do
        pcall(function() player:Destroy() end)
    end
    SyncState.remotePlayers = {}

    for id, vehicle in pairs(SyncState.remoteVehicles) do
        pcall(function() vehicle:Destroy() end)
    end
    SyncState.remoteVehicles = {}
end

LevelFuncs.OnUseItem = function(item)
    if not ConnectionState.isConnected then
        LevelFuncs.Engine.connect()
    end
end

LevelFuncs.OnFreeze = function()
    if ConnectionState.isConnected and ConnectionState.client then
        pcall(function()
            ConnectionState.client:send("FREEZE:" .. (ConnectionState.myId or 0) .. "\n")
        end)
    end
end

LevelFuncs.OnDoubleSoundEffect = function(doubleName, soundID)
    soundID = tonumber(soundID)
    if not soundID or soundID <= 0 then return end

    local double = TEN.Objects.GetMoveableByName(doubleName)
    if double then
        TEN.Sound.StopSound(soundID)
        local pos = double:GetPosition()
        TEN.Sound.PlaySound(soundID, pos)
    end
end

LevelFuncs.OnLaraSoundEffect = function(soundID, frameNumber)
    soundID = tonumber(soundID)
    frameNumber = tonumber(frameNumber)
    if not soundID or soundID <= 0 then return end

    if frameNumber ~= SoundState.lastSoundFrame[soundID] then
        SoundState.pendingSoundID = soundID
        SoundState.lastSoundFrame[soundID] = frameNumber
    end
end
